const { sendErrorResponse, sendResponse } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");
const fs = require("fs");
const path = require("path");
const { at, create } = require("lodash");
const e = require("express");

// CREATE Org_Position
module.exports.createPosition = async (req, res) => {
  const uploadedFiles = [];
  try {
    const {
      position_id_external,
      external_name_US,
      external_name_SA,
      entity_id,
      business_unit_id,
      division_id,
      department_id,
      section_id,
      change_reason_id,
      job_classification_id,
      job_level,
      employee_class_id,
      grade_id,
      target_fte,
      is_vacant = 1,
      location_id,
      cost_center_id,
      multiple_incubment_allowed = 1,
      position_criticality = 1,
      standard_hours,
      cust_country_id,
      cust_employment_type_id,
      cust_min,
      cust_max,
      cust_mid,
      parent_position_id,
      effective_start_date,
      status = 1,
    } = req.body;

    // Required fields check
    if (
      !position_id_external ||
      !external_name_US ||
      !entity_id ||
      !business_unit_id ||
      !division_id ||
      !department_id ||
      !section_id ||
      !job_classification_id ||
      !employee_class_id ||
      !grade_id ||
      !location_id ||
      !cost_center_id ||
      !standard_hours ||
      !cust_country_id ||
      !cust_employment_type_id ||
      !effective_start_date
    ) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Foreign key validation
    const [row] = await performQuery(
      `
  SELECT
    (? IS NULL OR EXISTS (
      SELECT 1 FROM ${tables.org_legal_entity} WHERE entity_id_external = ? AND is_deleted = 2
    )) AS legal_entity_ok,

    (? IS NULL OR EXISTS (
      SELECT 1 FROM ${tables.org_business_unit} WHERE bu_id_external = ? AND is_deleted = 2
    )) AS business_unit_ok,

    (? IS NULL OR EXISTS (
      SELECT 1 FROM ${tables.org_division} WHERE division_id_external = ? AND is_deleted = 2
    )) AS division_ok,

    (? IS NULL OR EXISTS (
      SELECT 1 FROM ${tables.org_department} WHERE department_id_external = ? AND is_deleted = 2
    )) AS department_ok,

    (? IS NULL OR EXISTS (
      SELECT 1 FROM ${tables.org_section} WHERE section_id_external = ? AND is_deleted = 2
    )) AS section_ok,

    (? IS NULL OR EXISTS (
      SELECT 1 FROM ${tables.picklist_master} WHERE id = ? AND is_deleted = 2
    )) AS change_reason_ok,

    (? IS NULL OR EXISTS (
      SELECT 1 FROM ${tables.org_job_classification} WHERE job_id_external = ? AND is_deleted = 2
    )) AS job_classification_ok,

    (? IS NULL OR EXISTS (
      SELECT 1 FROM ${tables.picklist_master} WHERE id = ? AND is_deleted = 2
    )) AS employee_class_ok,

    (? IS NULL OR EXISTS (
      SELECT 1 FROM ${tables.org_grade} WHERE grade_id_external = ? AND is_deleted = 2
    )) AS grade_ok,

    (? IS NULL OR EXISTS (
      SELECT 1 FROM ${tables.org_location} WHERE location_id_external = ? AND is_deleted = 2
    )) AS location_ok,

    (? IS NULL OR EXISTS (
      SELECT 1 FROM ${tables.org_cost_center} WHERE cost_center_id_external = ? AND is_deleted = 2
    )) AS cost_center_ok,

    (? IS NULL OR EXISTS (
      SELECT 1 FROM ${tables.org_position} WHERE position_id_external = ? AND is_deleted = 2
    )) AS parent_position_ok
  `,
      [
        // each field appears twice: for `? IS NULL` and for `id = ?`
        entity_id,
        entity_id,
        business_unit_id,
        business_unit_id,
        division_id,
        division_id,
        department_id,
        department_id,
        section_id,
        section_id,
        change_reason_id,
        change_reason_id,
        job_classification_id,
        job_classification_id,
        employee_class_id,
        employee_class_id,
        grade_id,
        grade_id,
        location_id,
        location_id,
        cost_center_id,
        cost_center_id,
        parent_position_id,
        parent_position_id,
      ]
    );
    if (
      !row.legal_entity_ok ||
      !row.business_unit_ok ||
      !row.division_ok ||
      !row.department_ok
    ) {
      const errorMessage =
        "One or more of Entity, Business Unit, Division, or Department is invalid";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    if (!row.section_ok && section_id) {
      const errorMessage = "Invalid Section";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    if (!row.change_reason_ok && change_reason_id) {
      const errorMessage = "Invalid Change Reason";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    if (!row.job_classification_ok) {
      const errorMessage = "Invalid Job Classification";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    if (!row.employee_class_ok && employee_class_id) {
      const errorMessage = "Invalid Employee Class";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    if (!row.grade_ok) {
      const errorMessage = "Invalid Grade";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    if (!row.location_ok) {
      const errorMessage = "Invalid Location";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    if (!row.cost_center_ok) {
      const errorMessage = "Invalid Cost Center";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    if (!row.parent_position_ok && parent_position_id) {
      const errorMessage = "Invalid Parent Position";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // File upload
    let baseURL = `${req.protocol}://${req.get("host")}/uploads`;
    const cust_attachment = req.files?.cust_attachment
      ? req.files.cust_attachment[0].filename
      : null;
    if (cust_attachment) uploadedFiles.push(cust_attachment);

    const cust_attachment_URL = cust_attachment
      ? `${baseURL}/${cust_attachment}`
      : null;

    // Duplicate position check
    const existing = await performQuery(
      `SELECT * FROM ${tables.org_position} WHERE position_id_external = ? and is_deleted = 2`,
      [position_id_external]
    );
    if (existing.length > 0) {
      const errorMessage = "Position record already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Get system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert query
    await performQuery(`INSERT INTO ${tables.org_position} SET ?`, {
      position_id_external,
      external_name_US,
      external_name_SA,
      entity_id,
      business_unit_id,
      division_id,
      department_id,
      section_id,
      change_reason_id,
      job_classification_id,
      job_level,
      employee_class_id,
      grade_id,
      target_fte,
      is_vacant,
      location_id,
      cost_center_id,
      multiple_incubment_allowed,
      position_criticality,
      standard_hours,
      cust_country_id,
      cust_employment_type_id,
      cust_min,
      cust_max,
      cust_mid,
      cust_attachment: cust_attachment_URL,
      parent_position_id,
      effective_start_date,
      status,
      created_at: currentTime,
      created_by: req?.user?.id,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    });

    return sendResponse(res, {}, "Position created successfully", 201);
  } catch (error) {
    console.error("Error while creating position:", error);
    return sendErrorResponse(res, error, "Error while creating position");
  }
};

// READ Org_postion (s)
module.exports.getPostion = async (req, res) => {
  try {
    const { id } = req.query;
    let query = `
      SELECT
      orgPosition.*,
      pr_change_reason.picklist_option AS change_reason_value,
      emp_class.picklist_option AS employee_class_value,
      legalEntity.name AS legalEntity_name,
      bu_unit.description AS bu_name,
      org_div.description AS div_name,
      org_dep.description AS dep_name,
      org_sec.description AS sec_name,
      org_job_cl.name_us AS job_classification_name,
      org_grad.grade AS grad_name,
      cost_center.cost_center_id_external, cost_center.name_us AS cost_center_name,
      location.location_id_external,
      location.location_name,
      location.address_line_1,
      location.address_line_2,
      location.address_line_3,
      location.locality,
      location.dependent_locality,
      location.administrative_area,
      location.sub_administrative_area,
      location.postal_code,
      location.country_code,
      location.sorting_code,
      location.language_code
      FROM ${tables.org_position} AS orgPosition
      LEFT JOIN ${tables.org_legal_entity} AS legalEntity ON legalEntity.entity_id_external = orgPosition.entity_id
      LEFT JOIN ${tables.org_business_unit} AS bu_unit ON bu_unit.bu_id_external = orgPosition.business_unit_id
      LEFT JOIN ${tables.org_division} AS org_div ON org_div.division_id_external = orgPosition.division_id
      LEFT JOIN ${tables.org_department} AS org_dep ON org_dep.department_id_external = orgPosition.department_id
      LEFT JOIN ${tables.org_section} AS org_sec ON org_sec.section_id_external = orgPosition.section_id
      LEFT JOIN ${tables.org_job_classification} AS org_job_cl ON org_job_cl.job_id_external = orgPosition.job_classification_id
      LEFT JOIN ${tables.org_grade} AS org_grad ON org_grad.grade_id_external = orgPosition.grade_id
      LEFT Join ${tables.picklist_master} AS pr_change_reason ON pr_change_reason.id = orgPosition.change_reason_id AND pr_change_reason.is_deleted = 2 AND pr_change_reason.picklist_id = 4
      LEFT Join ${tables.picklist_master} AS emp_class ON emp_class.id = orgPosition.employee_class_id AND emp_class.is_deleted = 2 AND emp_class.picklist_id = 3
      LEFT JOIN ${tables.org_cost_center} AS cost_center ON cost_center.cost_center_id_external = orgPosition.cost_center_id
      LEFT JOIN ${tables.org_location} AS location ON location.location_id_external = orgPosition.location_id
      WHERE orgPosition.is_deleted = 2
    `;

    if (id) query += ` AND id = ${id}`;
    query += ` ORDER BY id DESC`;

    const result = await performQuery(query);
    return sendResponse(res, result, "Position fetched successfully", 200);
  } catch (error) {
    console.error("Error While Fetching Postion: ", error);
    return sendErrorResponse(res, error, "Error while fetching position");
  }
};

// UPDATE Org_Position
module.exports.updatePosition = async (req, res) => {
  const uploadedFiles = [];
  try {
    const {
      id,
      external_name_US,
      external_name_SA,
      entity_id,
      business_unit_id,
      division_id,
      department_id,
      section_id,
      change_reason_id,
      job_classification_id,
      job_level,
      employee_class_id,
      grade_id,
      target_fte,
      is_vacant = 1,
      location_id,
      cost_center_id,
      multiple_incubment_allowed = 1,
      position_criticality = 1,
      standard_hours,
      cust_country_id,
      cust_employment_type_id,
      cust_min,
      cust_max,
      cust_mid,
      parent_position_id,
      effective_start_date,
      status = 1,
    } = req.body;

    // Required fields check
    if (
      !id ||
      !external_name_US ||
      !external_name_SA ||
      !entity_id ||
      !business_unit_id ||
      !division_id ||
      !department_id ||
      !section_id ||
      !job_classification_id ||
      !employee_class_id ||
      !grade_id ||
      !location_id ||
      !cost_center_id ||
      !standard_hours ||
      !cust_country_id ||
      !cust_employment_type_id ||
      !effective_start_date
    ) {
      const errorMessage =
        "Missing required fields: id, external_name_US, external_name_SA, entity_id, business_unit_id, division_id, department_id, section_id, job_classification_id, employee_class_id, grade_id, location_id, cost_center_id, standard_hours, cust_country_id, cust_employment_type_id, effective_start_date";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Handle file upload
    let baseURL = `${req.protocol}://${req.get("host")}/uploads`;
    const cust_attachment = req.files?.cust_attachment
      ? req.files.cust_attachment[0].filename
      : null;
    if (cust_attachment) uploadedFiles.push(cust_attachment);

    // Check Record Exists
    const existing = await performQuery(
      `SELECT * FROM ${tables.org_position} WHERE id = ? AND is_deleted = 2`,
      [id]
    );

    if (!existing || existing.length === 0) {
      const errorMessage = "Position record not found";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    const oldRecord = existing[0];

    // Foreign key validation helper
    const validateForeignKey = async (table, id) => {
      if (!id) return false;
      const result = await performQuery(
        `SELECT id FROM ${table} WHERE id = ? AND is_deleted = 2`,
        [id]
      );
      return result.length > 0;
    };


    // Foreign key validation
    const [row] = await performQuery(
    `
      SELECT
        (? IS NULL OR EXISTS (
          SELECT 1 FROM ${tables.org_legal_entity} WHERE entity_id_external = ? AND is_deleted = 2
        )) AS legal_entity_ok,

        (? IS NULL OR EXISTS (
          SELECT 1 FROM ${tables.org_business_unit} WHERE bu_id_external = ? AND is_deleted = 2
        )) AS business_unit_ok,

        (? IS NULL OR EXISTS (
          SELECT 1 FROM ${tables.org_division} WHERE division_id_external = ? AND is_deleted = 2
        )) AS division_ok,

        (? IS NULL OR EXISTS (
          SELECT 1 FROM ${tables.org_department} WHERE department_id_external = ? AND is_deleted = 2
        )) AS department_ok,

        (? IS NULL OR EXISTS (
          SELECT 1 FROM ${tables.org_section} WHERE section_id_external = ? AND is_deleted = 2
        )) AS section_ok,

        (? IS NULL OR EXISTS (
          SELECT 1 FROM ${tables.picklist_master} WHERE id = ? AND is_deleted = 2
        )) AS change_reason_ok,

        (? IS NULL OR EXISTS (
          SELECT 1 FROM ${tables.org_job_classification} WHERE job_id_external = ? AND is_deleted = 2
        )) AS job_classification_ok,

        (? IS NULL OR EXISTS (
          SELECT 1 FROM ${tables.picklist_master} WHERE id = ? AND is_deleted = 2
        )) AS employee_class_ok,

        (? IS NULL OR EXISTS (
          SELECT 1 FROM ${tables.org_grade} WHERE grade_id_external = ? AND is_deleted = 2
        )) AS grade_ok,

        (? IS NULL OR EXISTS (
          SELECT 1 FROM ${tables.org_location} WHERE location_id_external = ? AND is_deleted = 2
        )) AS location_ok,

        (? IS NULL OR EXISTS (
          SELECT 1 FROM ${tables.org_cost_center} WHERE cost_center_id_external = ? AND is_deleted = 2
        )) AS cost_center_ok,

        (? IS NULL OR EXISTS (
          SELECT 1 FROM ${tables.org_position} WHERE position_id_external = ? AND is_deleted = 2
        )) AS parent_position_ok
    `,
    [
      // each field appears twice: for `? IS NULL` and for `id = ?`
      entity_id,
      entity_id,
      business_unit_id,
      business_unit_id,
      division_id,
      division_id,
      department_id,
      department_id,
      section_id,
      section_id,
      change_reason_id,
      change_reason_id,
      job_classification_id,
      job_classification_id,
      employee_class_id,
      employee_class_id,
      grade_id,
      grade_id,
      location_id,
      location_id,
      cost_center_id,
      cost_center_id,
      parent_position_id,
      parent_position_id,
    ]
    );
    if (
      !row.legal_entity_ok ||
      !row.business_unit_ok ||
      !row.division_ok ||
      !row.department_ok
    ) {
      const errorMessage =
        "One or more of Entity, Business Unit, Division, or Department is invalid";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    if (!row.section_ok && section_id) {
      const errorMessage = "Invalid Section";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    if (!row.change_reason_ok && change_reason_id) {
      const errorMessage = "Invalid Change Reason";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    if (!row.job_classification_ok) {
      const errorMessage = "Invalid Job Classification";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    if (!row.employee_class_ok && employee_class_id) {
      const errorMessage = "Invalid Employee Class";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    if (!row.grade_ok) {
      const errorMessage = "Invalid Grade";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    if (!row.location_ok) {
      const errorMessage = "Invalid Location";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    if (!row.cost_center_ok) {
      const errorMessage = "Invalid Cost Center";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    if (!row.parent_position_ok && parent_position_id) {
      const errorMessage = "Invalid Parent Position";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Duplicate position check
    const duplicate = await performQuery(
      `SELECT * FROM ${tables.org_position} WHERE position_id_external = ? and id<>? and is_deleted = 2`,
      [oldRecord.position_id_external, id]
    );
    if (duplicate.length > 0) {
      const errorMessage = "Position record already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Prepare update object
    const updatedData = {
      position_id_external: oldRecord.position_id_external,
      sequence_no: Number(oldRecord.sequence_no) + 1,
      external_name_US: external_name_US || oldRecord.external_name_US,
      external_name_SA: external_name_SA || oldRecord.external_name_SA,
      entity_id: entity_id || oldRecord.entity_id,
      business_unit_id: business_unit_id || oldRecord.business_unit_id,
      division_id: division_id || oldRecord.division_id,
      department_id: department_id || oldRecord.department_id,
      section_id: section_id || oldRecord.section_id,
      change_reason_id: change_reason_id || oldRecord.change_reason_id,
      job_classification_id:
        job_classification_id || oldRecord.job_classification_id,
      job_level: job_level || oldRecord.job_level,
      employee_class_id: employee_class_id || oldRecord.employee_class_id,
      grade_id: grade_id || oldRecord.grade_id,
      target_fte: target_fte || oldRecord.target_fte,
      is_vacant: is_vacant || oldRecord.is_vacant,
      location_id: location_id || oldRecord.location_id,
      cost_center_id: cost_center_id || oldRecord.cost_center_id,
      multiple_incubment_allowed:
        multiple_incubment_allowed || oldRecord.multiple_incubment_allowed,
      position_criticality:
        position_criticality || oldRecord.position_criticality,
      standard_hours: standard_hours || oldRecord.standard_hours,
      cust_country_id: cust_country_id || oldRecord.cust_country_id,
      cust_employment_type_id:
        cust_employment_type_id || oldRecord.cust_employment_type_id,
      cust_min: cust_min || oldRecord.cust_min,
      cust_mid: cust_mid || oldRecord.cust_mid,
      cust_max: cust_max || oldRecord.cust_max,
      parent_position_id: parent_position_id || oldRecord.parent_position_id,
      effective_start_date:
        effective_start_date || oldRecord.effective_start_date,
      status: status || oldRecord.status,
      cust_attachment: cust_attachment
        ? `${baseURL}/${cust_attachment}`
        : oldRecord.cust_attachment,
      created_at: currentTime,
      created_by: req?.user?.id || null,
      updated_at: currentTime,
      updated_by: req?.user?.id || null,
    };

    // Update query to maintain history
    await performQuery(`UPDATE ${tables.org_position} SET ? WHERE id = ?`, [{
      is_deleted: 1,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    }, id]);

    // Insert new record with updated data
    await performQuery(`INSERT INTO ${tables.org_position} SET ?`, updatedData)

    return sendResponse(res, {}, "Position updated successfully", 200);
  } catch (error) {
    console.error("Error While Updating Position: ", error);
    return sendErrorResponse(res, error, "Error while updating Position");
  }
};
