const { sendErrorResponse, sendResponse } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");

// CREATE Org Unit
module.exports.createLegalEntity = async (req, res) => {
  try {
    const {
      entity_id_external,
      effective_start_date,
      org_registration_date,
      org_unit_code,
      name,
      description,
      description_arabic,
      default_pay_group_external_code,
      default_location_external_code,
      standard_weekly_hours,
      currency_code,
      country_of_registeration_code,
      name_format_external_code,
      display_name_format_external_code,
    } = req.body;

    // Validation
    if (!entity_id_external || !effective_start_date || !org_registration_date || !org_unit_code || !name) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    if(default_pay_group_external_code){
      // check is pay_group_external_code exists in pay_group table
      const isPayGroupExists = await performQuery(
        `SELECT id FROM ${tables.org_pay_group} WHERE id = ? AND is_deleted = 2`,
        [default_pay_group_external_code]
      );
      if (isPayGroupExists.length === 0) {
        const errorMessage = "Default Pay Group External Code not found in Pay Group table";
        return sendErrorResponse(res, errorMessage, errorMessage);
      }
    }

    if(default_location_external_code){
      // check is location_external_code exists in location table
      const isLocationExists = await performQuery(
        `SELECT id FROM ${tables.org_location} WHERE id = ? AND is_deleted = 2`,
        [default_location_external_code]
      );
      if (isLocationExists.length === 0) {
        const errorMessage = "Default Location External Code not found in Location table";
        return sendErrorResponse(res, errorMessage, errorMessage);
      }
    }

    // Check for duplicate Org Unit Code
    const duplicateCheck = await performQuery(
      `SELECT id FROM ${tables.org_legal_entity} WHERE (entity_id_external = ? OR org_unit_code = ?) AND is_deleted = 2`,
      [entity_id_external, org_unit_code]
    );

    if (duplicateCheck && duplicateCheck.length > 0) {
      const errorMessage = "Organization unit code already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert record
    await performQuery(`INSERT INTO ${tables.org_legal_entity} SET ?`, {
      sequence_no : 1,
      entity_id_external,
      effective_start_date,
      org_registration_date,
      org_unit_code,
      name,
      description,
      description_arabic,
      default_pay_group_external_code,
      default_location_external_code,
      standard_weekly_hours,
      currency_code,
      country_of_registeration_code,
      name_format_external_code,
      display_name_format_external_code,
      created_at: currentTime,
      created_by: req?.user?.id,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    });

    return sendResponse(res, {}, "Organization Unit created successfully", 201);
  } catch (error) {
    console.error("Error While Creating Organization Unit: ", error);
    return sendErrorResponse(
      res,
      error,
      "Error while creating organization unit"
    );
  }
};

// READ Org Unit(s)
module.exports.getLegalEntity = async (req, res) => {
  try {
    const {
      id,
       entity_id_external, 
       org_unit_code
    } = req.query;

    let query = `
      SELECT 
        le.*, 
        loc.location_name, loc.effective_start_date AS location_effective_start_date, loc.address_line_1, loc.address_line_2, loc.address_line_3,
        loc.locality, loc.dependent_locality, loc.administrative_area, loc.sub_administrative_area,
        loc.postal_code, loc.country_code, loc.sorting_code, loc.language_code,
        lg.location_group_name, lg.effective_start_date AS location_group_effective_start_date,
        pg.pay_group_id_external, pg.group_name, pg.effective_start_date AS pay_group_effective_start_date
      FROM ${tables.org_legal_entity} le
      LEFT JOIN ${tables.org_location} loc ON le.default_location_external_code = loc.id AND loc.is_deleted = 2
      LEFT JOIN ${tables.org_location_group} lg ON loc.location_group_id = lg.id AND lg.is_deleted = 2
      LEFT JOIN ${tables.org_pay_group} pg ON le.default_pay_group_external_code = pg.id AND pg.is_deleted = 2

      WHERE le.is_deleted = 2
    `;

    if (id) query += ` AND le.id = ${id}`;
    if (entity_id_external) query += ` AND le.entity_id_external = '${entity_id_external}'`;
    if (org_unit_code) query += ` AND le.org_unit_code = '${org_unit_code}'`;

    query += ` ORDER BY id DESC`;

    const result = await performQuery(query);
    return sendResponse(
      res,
      result,
      "Organization Unit(s) fetched successfully",
      200
    );
  } catch (error) {
    console.error("Error While Fetching Organization Unit(s): ", error);
    return sendErrorResponse(
      res,
      error,
      "Error while fetching organization unit(s)"
    );
  }
};

// UPDATE Org Unit
module.exports.updateLegalEntity = async (req, res) => {
  try {
    const { 
      id,
      name,
      effective_start_date,
      org_registration_date,
      org_unit_code,
      description,
      description_arabic,
      status,
      default_pay_group_external_code,
      default_location_external_code,
      standard_weekly_hours,
      currency_code,
      country_of_registeration_code,
      name_format_external_code,
      display_name_format_external_code,
    } = req.body;

    if (!id || !effective_start_date || !org_registration_date || !org_unit_code || !name) {
      const errorMessage = "Missing Required Fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Fetch existing record
    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.org_legal_entity} WHERE id = ? AND is_deleted = 2`,
      [id]
    );
    if (!existingRecord || existingRecord.length === 0) {
      const errorMessage = "Organization Unit not found";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    const oldRecord = existingRecord[0];

    // Duplicate check for org_unit_code (excluding self)
    const duplicateCheck = await performQuery(
      `SELECT id FROM ${tables.org_legal_entity} WHERE (entity_id_external = ? OR org_unit_code = ?) AND id <> ? AND is_deleted = 2`,
      [oldRecord.entity_id_external, org_unit_code, id]
    );
    if (duplicateCheck && duplicateCheck.length > 0) {
      const errorMessage = "Organization unit code already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    if(default_pay_group_external_code){
      // check is pay_group_external_code exists in pay_group table
      const isPayGroupExists = await performQuery(
        `SELECT id FROM ${tables.org_pay_group} WHERE pay_group_id_external = ? AND is_deleted = 2`,
        [default_pay_group_external_code]
      );
      if (isPayGroupExists.length === 0) {
        const errorMessage = "Default Pay Group External Code not found in Pay Group table";
        return sendErrorResponse(res, errorMessage, errorMessage);
      }
    }

    if(default_location_external_code){
      // check is location_external_code exists in location table
      const isLocationExists = await performQuery(
        `SELECT id FROM ${tables.org_location} WHERE location_id_external = ? AND is_deleted = 2`,
        [default_location_external_code]
      );
      if (isLocationExists.length === 0) {
        const errorMessage = "Default Location External Code not found in Location table";
        return sendErrorResponse(res, errorMessage, errorMessage);
      }
    }


    // System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Deactivate existing record to maintain history
        await performQuery(
            `UPDATE ${tables.org_legal_entity} SET ? WHERE id = ?`,
            [{
                is_deleted: 1,
                updated_by: req?.user?.id,
                updated_at: currentTime
            },id]
        );

    // Insert updated record
    await performQuery(`INSERT INTO ${tables.org_legal_entity} SET ?`, [
      {
        sequence_no: Number(oldRecord.sequence_no) + 1,
        entity_id_external: oldRecord.entity_id_external,
        name: name,
        effective_start_date: effective_start_date || oldRecord.effective_start_date,
        org_registration_date: org_registration_date || oldRecord.org_registration_date,
        org_unit_code: org_unit_code || oldRecord.org_unit_code,
        description: description || oldRecord.description,
        description_arabic: description_arabic || oldRecord.description_arabic,
        status: status || oldRecord.status,
        default_pay_group_external_code: default_pay_group_external_code || oldRecord.default_pay_group_external_code,
        default_location_external_code: default_location_external_code || oldRecord.default_location_external_code,
        standard_weekly_hours: standard_weekly_hours || oldRecord.standard_weekly_hours,
        currency_code: currency_code || oldRecord.currency_code,
        country_of_registeration_code: country_of_registeration_code || oldRecord.country_of_registeration_code,
        name_format_external_code: name_format_external_code || oldRecord.name_format_external_code,
        display_name_format_external_code: display_name_format_external_code || oldRecord.display_name_format_external_code,
        created_at: currentTime,
        created_by: req?.user?.id,
        updated_at: currentTime,
        updated_by: req?.user?.id,
      },
      id
    ]);

    return sendResponse(res, {}, "Organization Unit updated successfully", 200);
  } catch (error) {
    console.error("Error While Updating Organization Unit: ", error);
    return sendErrorResponse(
      res,
      error,
      "Error while updating organization unit"
    );
  }
};

// DELETE Org Unit (Soft Delete)
module.exports.deleteLegalEntity = async (req, res) => {
  try {
    const { id } = req.query;

    if (!id) {
      const errorMessage = "Organization Unit ID is required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    await performQuery(`UPDATE ${tables.org_legal_entity} SET ? WHERE id = ?`, [
      {
        is_deleted: 1,
        updated_at: currentTime,
        updated_by: req?.user?.id,
      },
      id,
    ]);

    return sendResponse(res, {}, "Organization Unit deleted successfully", 200);
  } catch (error) {
    console.error("Error While Deleting Organization Unit: ", error);
    return sendErrorResponse(
      res,
      error,
      "Error while deleting organization unit"
    );
  }
};
