const { sendErrorResponse, sendResponse, } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");

// --> Create Work Schedule Details
module.exports.createWorkScheduleDetails = async (req, res) => {
    try {
    const {
      schedule_id_external,
      day,
      day_model_id_external,
      day_model_category
    } = req.body;

    if (
      !schedule_id_external ||
      !day ||
      !day_model_category
    ) {
      const errorMessage = "Missing required fields!";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Duplicate check
    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.work_schedule_details}
       WHERE schedule_id_external = ? AND day = ? AND is_active = 1`,
      [schedule_id_external, day]
    );
    if (existingRecord.length > 0) {
      return sendErrorResponse(
        res,
        "Work schedule detail already exists for this day",
        "Work schedule detail already exists for this day"
      );
    }

    // Picklist check
    const checkCategory = await performQuery(
      `SELECT * FROM ${tables.picklist_master}
       WHERE picklist_id = 21 AND id = ? AND is_deleted = 2`,
      [day_model_category]
    );
    if (checkCategory.length === 0) {
      return sendErrorResponse(res, "Invalid day model category", "Invalid day model category");
    }

    // check for valid schedule_id_external
    const scheduleRecord = await performQuery(
      `SELECT * FROM ${tables.work_schedule}
        WHERE schedule_id_external = ? AND is_active = 1`,  
      [schedule_id_external]
    );
    if (scheduleRecord.length === 0) {
      return sendErrorResponse(res, "Invalid schedule_id_external", "Invalid schedule_id_external");
    }

    if(day_model_id_external){
      // Check for valid day_model_id_external
      const dayModelRecord = await performQuery(
        `SELECT * FROM ${tables.day_model} WHERE day_model_id_external = ? AND is_active = 1`,
        [day_model_id_external]
      );
      if (dayModelRecord.length === 0) {
        return sendErrorResponse(res, "Invalid day_model_id_external", "Invalid day_model_id_external");
      }
    }

    // Get current system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert new record
    await performQuery(
      `INSERT INTO ${tables.work_schedule_details} SET ?`,
      {
        sequence_no: 1,
        schedule_id_external,
        day,
        day_model_id_external,
        day_model_category,
        created_by: req?.user?.id,
        created_at: currentTime,
        updated_by: req?.user?.id,
        updated_at: currentTime
      }
    );

    return sendResponse(res, {}, "Work Schedule Detail created successfully", 200);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while creating Work Schedule Detail");
  }
}


// --> Get Work Schedule Details
module.exports.getWorkSchedulesDetails = async (req, res) => {
    try {
    const { id, schedule_id_external } = req.query;

    let query = `
      SELECT 
        wsd.id,
        wsd.sequence_no,
        wsd.schedule_id_external,
        ws.external_name AS schedule_external_name,
        wsd.day,
        wsd.day_model_id_external,
        dm.external_name AS day_model_external_name,
        wsd.day_model_category,
        pm.picklist_option AS day_model_category_value
      FROM ${tables.work_schedule_details} wsd
      LEFT JOIN ${tables.picklist_master} pm
        ON wsd.day_model_category = pm.id
        AND pm.picklist_id = 21
        AND pm.is_deleted = 2
      LEFT JOIN ${tables.work_schedule} ws
        ON wsd.schedule_id_external = ws.schedule_id_external
        AND ws.is_active = 1
      LEFT JOIN ${tables.day_model} dm
        ON wsd.day_model_id_external = dm.day_model_id_external
        AND dm.is_active = 1
      WHERE wsd.is_active = 1
    `;

    if (id) query += ` AND wsd.id = ${id}`;
    if (schedule_id_external) query += ` AND wsd.schedule_id_external = '${schedule_id_external}'`;

    query += ` ORDER BY wsd.id DESC`;

    const result = await performQuery(query);
    return sendResponse(res, result, "Work Schedule Detail fetched successfully", 200);

  } catch (error) {
    return sendErrorResponse(res, error, "Error While Fetching Work Schedule Detail");
  }
}


// --> Update Work Schedule Details
module.exports.updateWorkScheduleDetails = async (req, res) => {
    try {
    const {
      id,
      day,
      day_model_id_external,
      day_model_category
    } = req.body;

    if (!id || !day || !day_model_id_external || !day_model_category) {
      const errorMessage = "Missing required fields!";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.work_schedule_details} WHERE id = ? AND is_active = 1`,
      [id]
    );
    if (existingRecord.length === 0) {
      return sendErrorResponse(res, "Work Schedule Detail not found", "Work Schedule Detail not found");
    }

    const schedule_id_external = existingRecord[0].schedule_id_external;

    // Duplicate day check
    const duplicateRecord = await performQuery(
      `SELECT * FROM ${tables.work_schedule_details}
       WHERE schedule_id_external = ?
       AND day = ?
       AND is_active = 1
       AND id <> ?`,
      [schedule_id_external, day, id]
    );
    if (duplicateRecord.length > 0) {
      return sendErrorResponse(
        res,
        "Work schedule detail already exists for this day",
        "Work schedule detail already exists for this day"
      );
    }

    // Picklist check
    const checkCategory = await performQuery(
      `SELECT * FROM ${tables.picklist_master}
       WHERE picklist_id = 21 AND id = ? AND is_deleted = 2`,
      [day_model_category]
    );
    if (checkCategory.length === 0) {
      return sendErrorResponse(res, "Invalid day model category", "Invalid day model category");
    }

    // Check for valid day_model_id_external
    const dayModelRecord = await performQuery(
      `SELECT * FROM ${tables.day_model} WHERE day_model_id_external = ? AND is_active = 1`,
      [day_model_id_external]
    );
    if (dayModelRecord.length === 0) {
      return sendErrorResponse(res, "Invalid day_model_id_external", "Invalid day_model_id_external");
    }

    // Get current system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Deactivate old record
    await performQuery(
      `UPDATE ${tables.work_schedule_details} SET ? WHERE id = ?`,
      [{
        is_active: 2,
        updated_by: req?.user?.id,
        updated_at: currentTime
      }, id]
    );

    // Insert new record with incremented sequence_no
    await performQuery(
      `INSERT INTO ${tables.work_schedule_details} SET ?`,
      {
        sequence_no: Number(existingRecord[0].sequence_no) + 1,
        schedule_id_external : existingRecord[0].schedule_id_external,
        day,
        day_model_id_external,
        day_model_category,
        created_by: req?.user?.id,
        created_at: currentTime,
        updated_by: req?.user?.id,
        updated_at: currentTime
      }
    );

    return sendResponse(res, {}, "Work Schedule Detail updated successfully", 200);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while updating Work Schedule Detail");
  }
}