const { sendErrorResponse, sendResponse, } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");

// --> Create Day Model
module.exports.createDayModel = async (req, res) => {
    try {
    const {
      day_model_id_external,
      external_name,
      planned_hours,
      planned_hours_formated,
      country,
      shift_classification,
      time_recording_variant,
      cross_midnight_allowed = 2,
      non_working_day = 2
    } = req.body;

    if (
      !day_model_id_external ||
      !external_name ||
      !planned_hours ||
      !planned_hours_formated ||
      !country ||
      !time_recording_variant || 
      !cross_midnight_allowed ||
      !non_working_day
    ) {
      const errorMessage = "Missing required fields!";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Duplicate check
    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.day_model}
       WHERE day_model_id_external = ? AND is_active = 1`,
      [day_model_id_external]
    );
    if (existingRecord.length > 0) {
      return sendErrorResponse(
        res,
        "Day Model already exists",
        "Day Model already exists"
      );
    }

    // Picklist check
    const checkVariant = await performQuery(
      `SELECT * FROM ${tables.picklist_master}
       WHERE picklist_id = 19 AND id = ? AND is_deleted = 2`,
      [time_recording_variant]
    );
    if (checkVariant.length === 0) {
      return sendErrorResponse(res, "Invalid time recording variant", "Invalid time recording variant");
    }

    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    await performQuery(
      `INSERT INTO ${tables.day_model} SET ?`,
      {
        sequence_no: 1,
        day_model_id_external,
        external_name,
        planned_hours,
        planned_hours_formated,
        country,
        shift_classification,
        time_recording_variant,
        cross_midnight_allowed,
        non_working_day,
        created_by: req?.user?.id,
        created_at: currentTime,
        updated_by: req?.user?.id,
        updated_at: currentTime
      }
    );

    return sendResponse(res, {}, "Day Model created successfully", 200);

  } catch (error) {
    console.error("Error While Creating Day Model", error);
    return sendErrorResponse(res, error, "Error while creating Day Model");
  }
};


// --> Get Day Model
module.exports.getDayModel = async (req, res) => {
    try {
    const { id, day_model_id_external } = req.query;

    let query = `
      SELECT
        dm.id,
        dm.sequence_no,
        dm.day_model_id_external,
        dm.external_name,
        dm.planned_hours,
        dm.planned_hours_formated,
        dm.country,
        dm.shift_classification,
        dm.time_recording_variant,
        pm.picklist_option AS time_recording_variant_value,
        dm.cross_midnight_allowed,
        dm.non_working_day,
        dms.id AS day_model_segment_id,
        dms.start_time, dms.end_time, dms.duration,
        dms.day_model_segments_category, pm_dm_segement_category.picklist_option AS day_model_segment_category_value,
        dms.sequence_no,
        dms.created_at AS day_model_segment_created_at,
        dms.updated_at AS day_model_segment_updated_at
      FROM ${tables.day_model} dm
      LEFT JOIN ${tables.picklist_master} pm ON dm.time_recording_variant = pm.id AND pm.picklist_id = 19 AND pm.is_deleted = 2
      LEFT JOIN ${tables.day_model_segments} dms ON dm.day_model_id_external = dms.day_model_id_external AND dms.is_active = 1
      LEFT JOIN ${tables.picklist_master} pm_dm_segement_category ON dms.day_model_segments_category = pm_dm_segement_category.id AND pm_dm_segement_category.picklist_id = 20 AND pm_dm_segement_category.is_deleted = 2
        
      WHERE dm.is_active = 1
    `;

    if (id) query += ` AND dm.id = ${id}`;
    if (day_model_id_external) query += ` AND dm.day_model_id_external = '${day_model_id_external}'`;

    query += ` ORDER BY dm.id DESC`;

    const result = await performQuery(query);
    const transformedResult = await TransformData(result);
    return sendResponse(res, transformedResult, "Day Model fetched successfully", 200);

  } catch (error) {
    console.error("Error While Fetching Day Model", error);
    return sendErrorResponse(res, error, "Error While Fetching Day Model");
  }
}

// --> Update Day Model 
module.exports.updateDayModel = async (req, res) => {
    try {
    const {
      id,
      external_name,
      planned_hours,
      planned_hours_formated,
      country,
      shift_classification,
      time_recording_variant,
      cross_midnight_allowed = 0,
      non_working_day = 0
    } = req.body;

    if (
      !id ||
      !external_name ||
      planned_hours === undefined ||
      !planned_hours_formated ||
      !country ||
      !time_recording_variant
    ) {
      const errorMessage = "Missing required fields!";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.day_model} WHERE id = ? AND is_active = 1`,
      [id]
    );
    if (existingRecord.length === 0) {
      return sendErrorResponse(res, "Day Model not found", "Day Model not found");
    }

    // Picklist check
    const checkVariant = await performQuery(
      `SELECT * FROM ${tables.picklist_master}
       WHERE picklist_id = 19 AND id = ? AND is_deleted = 2`,
      [time_recording_variant]
    );
    if (checkVariant.length === 0) {
      return sendErrorResponse(res, "Invalid time recording variant", "Invalid time recording variant");
    }

    // Check for duplicate on external_name in same country (excluding current record)
    const duplicateCheck = await performQuery(
      `SELECT * FROM ${tables.day_model} WHERE external_name = ? AND country = ? AND id <> ?`,
      [external_name, country, id]
    );
    if (duplicateCheck.length > 0) {
      return sendErrorResponse(res, "Duplicate external_name in the same country", "Duplicate external_name in the same country");
    }

    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Deactivate old record
    await performQuery(
      `UPDATE ${tables.day_model} SET ? WHERE id = ?`,
      [{
        is_active: 2,
        updated_by: req?.user?.id,
        updated_at: currentTime
      }, id]
    );

    // Insert new record
    await performQuery(
      `INSERT INTO ${tables.day_model} SET ?`,
      {
        sequence_no: Number(existingRecord[0].sequence_no) + 1,
        day_model_id_external: existingRecord[0].day_model_id_external,
        external_name,
        planned_hours,
        planned_hours_formated,
        country,
        shift_classification,
        time_recording_variant,
        cross_midnight_allowed,
        non_working_day,
        created_by: req?.user?.id,
        created_at: currentTime,
        updated_by: req?.user?.id,
        updated_at: currentTime
      }
    );

    return sendResponse(res, {}, "Day Model updated successfully", 200);

  } catch (error) {
    console.error("Error While Updating Day Model", error);
    return sendErrorResponse(res, error, "Error while updating Day Model");
  }
}


const TransformData = async (data) => {
  const map = new Map();
  data?.forEach((row) => {
    // Grouping key: adjust if you want a different grouping logic
    const key = `${row.id}`;
    if (!map.has(key)) {
      map.set(key, {
        id: row.id,
        day_model_id_external: row.day_model_id_external,
        external_name: row.external_name,
        planned_hours: row.planned_hours,
        planned_hours_formated: row.planned_hours_formated,
        country: row.country,
        shift_classification: row.shift_classification,
        time_recording_variant_value: row.time_recording_variant_value,
        cross_midnight_allowed: row.cross_midnight_allowed,
        non_working_day: row.non_working_day,
        created_at: row.created_at,
        updated_at: row.updated_at,
        created_by: row.created_by,
        updated_by: row.updated_by,
        day_model_segments: [],
      });
    }
    const agg = map.get(key);
    if (row.day_model_segment_id !== null) {
      const existingDayModelSegmentId = agg.day_model_segments.some(item => item.id === row.day_model_segment_id);
      if (!existingDayModelSegmentId) {
        agg.day_model_segments.push({
          id: row.day_model_segment_id,
          day_model_segment_name: row.day_model_segment_name,
          start_time: row.start_time,
          end_time: row.end_time,
          duration: row.duration,
          day_model_segments_category: row.day_model_segments_category,
          day_model_segment_category_value: row.day_model_segment_category_value,
        });
      }
    }
  });
  return Array.from(map.values());
};