const { isUndefined, create } = require("lodash");
const { sendErrorResponse, sendResponse } = require("../../utils");
const { performQuery } = require("../../utils/db");
const { tables } = require("../../utils/tables");
const { getSystemTime } = require("../../functions/getTimezone");
const moment = require("moment-timezone");
const fs = require('fs');
const path = require('path');


// API to Create National ID Info --> POST /api/personRecord/national-id
module.exports.createNationalIdInfo = async (req, res) => {
    const uploadedFiles = [];
    try {
        // Get Variable from req Body
        const {
            person_id, country, national_id_no, id_type, issue_date,
            issue_date_hijri, expiry_date, expiry_date_hijri
        } = req.body;

        // Validate Input Data
        if (
            !person_id || !country || !national_id_no || !id_type || !issue_date || !expiry_date
        ) {
            return sendErrorResponse(res, "Required fields are missing", "Validation Error", 400);
        }

        // Handle file uploads
        let baseURL = `${req.protocol}://${req.get("host")}/uploads`;
        const frontPic = req.files?.attachment_1 ? req.files.attachment_1[0].filename : null;
        const backPic = req.files?.attachment_2 ? req.files.attachment_2[0].filename : null;
        if (frontPic) uploadedFiles.push(frontPic);
        if (backPic) uploadedFiles.push(backPic);

        // send Error response if either upload file is missing
        if (!frontPic || !backPic){
            return sendErrorResponse(res, "Required files are missing", "Front & Back Picture of ID card is required", 400);
        }

        // check if person exists
        const personRecord = await performQuery(
            `Select * from ${tables.per_person} Where id = ? AND is_deleted = 2`,
            [person_id]
        );
        if (personRecord.length === 0) {
            return sendErrorResponse(res, "Person not found", "Person not Fount", 404);
        }

        // check for existing record against person_id or national_id_no
        const existingNationalId = await performQuery(
            `Select * from ${tables.per_national_id} Where (person_id = ? OR national_id_no = ?) AND is_deleted = 2`,
            [person_id, national_id_no]
        );
        if (existingNationalId.length > 0) {
            return sendErrorResponse(res, "National ID info already exists for this person", "Duplicate Record", 409);
        }

        // Calculate completion and is_completed
        let total_fields = 10; // Total fields to consider for completion
        let filled_fields = 0;

        if (person_id) filled_fields++;
        if (country) filled_fields++;
        if (national_id_no) filled_fields++;
        if (id_type) filled_fields++;
        if (issue_date) filled_fields++;
        if (issue_date_hijri) filled_fields++;
        if (expiry_date) filled_fields++;
        if (expiry_date_hijri) filled_fields++;
        if (frontPic) filled_fields++;
        if (backPic) filled_fields++;

        const completion = Math.round((filled_fields / total_fields) * 100);
        const is_completed = completion === 100 ? 2 : 1; // 2 for completed, 1 for not completed

        // Get System Timestamp
        const systemTime = await getSystemTime();
        const currentTime = systemTime ? moment(systemTime).format("YYYY-MM-DD HH:mm:ss") : null;
        
        // create File URLs
        const frontPicURL = frontPic ? `${baseURL}/${frontPic}` : null;
        const backPicURL = backPic ? `${baseURL}/${backPic}` : null;

        // Insert National ID Info
        const nationalID = await performQuery(
            `INSERT INTO ${tables.per_national_id} SET ?`,
            {
                person_id: person_id,
                country: country,
                national_id_no: national_id_no,
                id_type: id_type,
                issue_date: issue_date,
                issue_date_hijri: issue_date_hijri,
                expiry_date: expiry_date,
                expiry_date_hijri: expiry_date_hijri,
                attachment_1: frontPicURL,
                attachment_2: backPicURL,
                completion: completion,
                is_completed: is_completed,
                created_at: currentTime,
                created_by: req.user?.id || null,
                updated_at: currentTime,
                updated_by: req.user?.id || null
            }
        );

        // Send Response
        return sendResponse(res, null, "National ID info created successfully", 201);

    } catch (error) {
        console.log("Error in Creating National ID Info: ", error);
        return sendErrorResponse(res, error, "Failed to create national ID info", 500);        
    }
}

// API to Get National ID Info by Person ID / Person ID External --> GET /api/personRecord/national-id?person_id=
module.exports.getNationalIdInfo = async (req, res) => {
    try {
        // Get Variable from req query
        const { person_id, person_id_external} = req.query;

        // Query for Select Query
        let query = `
            SELECT pn.*, p.salutation, p.first_name, p.mid_name, p.last_name, p.display_name, p.person_id_external
            FROM ${tables.per_national_id} AS pn
            LEFT JOIN ${tables.per_person} AS p ON pn.person_id = p.id
            WHERE pn.is_deleted = 2`;

        // Apply Filters
        if (!isUndefined(person_id)) {
            query += ` AND pn.person_id = ${person_id}`;
        } else if (!isUndefined(person_id_external)) {
            query += ` AND p.person_id_external = '${person_id_external}'`;
        }

        // Sort by id in descending order
        query += ` ORDER BY pn.id DESC`;

        // Fetch National ID Info
        const nationalIdInfo = await performQuery(query);

        return sendResponse(res, nationalIdInfo, "National ID Selected successfully", 200);

    } catch (error) {
        console.log("Error in Getting National ID Info: ", error);
        return sendErrorResponse(res, error, "Failed to get national ID info", 500);
    }
}

 // API to Update National ID Info --> PUT /api/personRecord/national-id
module.exports.updateNationalIdInfo = async (req, res) => {
    const uploadedFiles = [];
    try {
        // Get Variable from req Body
        const {
            id, country, national_id_no, id_type, issue_date,
            issue_date_hijri, expiry_date, expiry_date_hijri
        } = req.body;

        // Validate Input Data
        if (
            !id || !country || !national_id_no || !id_type || !issue_date || !expiry_date
        ) {
            return sendErrorResponse(res, "Required fields are missing", "Validation Error", 400);
        }

        // Handle file uploads
        let baseURL = `${req.protocol}://${req.get("host")}/uploads`;
        const frontPic = req.files?.attachment_1 ? req.files.attachment_1[0].filename : null;
        const backPic = req.files?.attachment_2 ? req.files.attachment_2[0].filename : null;
        if (frontPic) uploadedFiles.push(frontPic);
        if (backPic) uploadedFiles.push(backPic);

        // Check is national_id_no already exists for other record
        const existingNationalIdCheck = await performQuery(
            `Select * from ${tables.per_national_id} Where national_id_no = ? AND id <> ? AND is_deleted = 2`,
            [national_id_no, id]
        );
        if (existingNationalIdCheck.length > 0) {
            return sendErrorResponse(res, "National ID number already exists for another record", "Duplicate Record", 409);
        }

        // get previous record
        const existingNationalId = await performQuery(
            `Select * from ${tables.per_national_id} Where id = ? AND is_deleted = 2`,
            [id]
        );
        if (existingNationalId.length === 0) {
            return sendErrorResponse(res, "National ID info not found", "Invalid Record", 404);
        }

        const previousRecord = existingNationalId[0];
        
        // Calculate completion and is_completed
        let total_fields = 10; // Total fields to consider for completion
        let filled_fields = 0;

        if (previousRecord.person_id) filled_fields++;
        if (country || previousRecord.country) filled_fields++;
        if (national_id_no || previousRecord.national_id_no) filled_fields++;
        if (id_type || previousRecord.id_type) filled_fields++;
        if (issue_date || previousRecord.issue_date) filled_fields++;
        if (issue_date_hijri || previousRecord.issue_date_hijri) filled_fields++;
        if (expiry_date || previousRecord.expiry_date) filled_fields++;
        if (expiry_date_hijri || previousRecord.expiry_date_hijri) filled_fields++;
        if (frontPic || previousRecord.attachment_1) filled_fields++;
        if (backPic || previousRecord.attachment_2) filled_fields++;

        const completion = Math.round((filled_fields / total_fields) * 100);
        const is_completed = completion === 100 ? 2 : 1; // 2 for completed, 1 for not completed

        // Get System Timestamp
        const systemTime = await getSystemTime();
        const currentTime = systemTime ? moment(systemTime).format("YYYY-MM-DD HH:mm:ss") : null;
        
        // create File URLs
        const frontPicURL = frontPic ? `${baseURL}/${frontPic}` : null;
        const backPicURL = backPic ? `${baseURL}/${backPic}` : null;

        // Update National ID Info
        const updateNationalID = await performQuery(
            `UPDATE ${tables.per_national_id} SET ? WHERE id = ?`,
            [{
                updated_at: currentTime,
                updated_by: req.user?.id || null,
                is_deleted: 1, // Mark previous record as deleted
            }, id]
        );

        // Insert New National ID Info as updated record
        const nationalID = await performQuery(
            `INSERT INTO ${tables.per_national_id} SET ?`,
            {
                person_id: previousRecord.person_id,
                country: country || previousRecord.country,
                national_id_no: national_id_no || previousRecord.national_id_no,
                id_type: id_type || previousRecord.id_type,
                issue_date: issue_date || previousRecord.issue_date,
                issue_date_hijri: issue_date_hijri || previousRecord.issue_date_hijri,
                expiry_date: expiry_date || previousRecord.expiry_date,
                expiry_date_hijri: expiry_date_hijri || previousRecord.expiry_date_hijri,
                attachment_1: frontPicURL || previousRecord.attachment_1,
                attachment_2: backPicURL || previousRecord.attachment_2,
                completion: completion,
                is_completed: is_completed,
                created_at: currentTime,
                created_by: req.user?.id || null,
                updated_at: currentTime,
                updated_by: req.user?.id || null,
            }
        );

        // Send Response
        return sendResponse(res, null, "National ID info updated successfully", 200);

    }
    catch (error) {
        console.log("Error in Updating National ID Info: ", error);
        return sendErrorResponse(res, error, "Failed to update national ID info", 500);
    }
}

// API to Delete National ID Info --> DELETE /api/personRecord/national-id?id=
module.exports.deleteNationalIdInfo = async (req, res) => {
    try {
        // get Variable from req query
        const { id } = req.query;
        // validate input
        if (isUndefined(id)) {
            return sendErrorResponse(res, "National ID info ID is required", "Validation Error", 400);
        }

        return sendErrorResponse(res, "National ID info deletion is not allowed", "Operation Not Allowed", 403);

        
    } catch (error) {
        console.log("Error in deleteNationalIdInfo:", error);
        return sendErrorResponse(res, error, "Failed to delete national ID info", 500);
    }
}