const { sendErrorResponse, sendResponse } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");

// CREATE Recurring Pay Component
module.exports.createRecurringPayComp = async (req, res) => {
  try {
    const {
      recurring_pay_comp_id_external,
      value,
      currency_code,
      start_date,
      frequency,
      sequence_no,
      enable_end_date,
      end_date,
      employee_id,
      note,
    } = req.body;

    // Validation
    if (
      !recurring_pay_comp_id_external || !value || !currency_code || 
      !start_date || !frequency || !employee_id ||
      (enable_end_date == 1 && !end_date)
    ) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check if employee exists
    // const isEmployee = await performQuery(
    //   `SELECT * FROM ${tables.employee} WHERE id = ? AND is_deleted = 2`,
    //   [employee_id]
    // );

    // if (isEmployee.length === 0) {
    //   const errorMessage = "Employee not found against employee_id";
    //   return sendErrorResponse(res, errorMessage, errorMessage);
    // }

    // Check duplicate recurring_pay_comp_id_external
    const duplicateCheck = await performQuery(
      `SELECT id FROM ${tables.org_recurring_pay_component} WHERE recurring_pay_comp_id_external = ? AND is_deleted = 2`,
      [recurring_pay_comp_id_external]
    );

    if (duplicateCheck && duplicateCheck.length > 0) {
      const errorMessage = "Recurring Pay Component external ID already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert record
    await performQuery(`INSERT INTO ${tables.org_recurring_pay_component} SET ?`, {
      sequence_no: 1,
      recurring_pay_comp_id_external,
      value,
      currency_code,
      start_date,
      frequency,
      sequence_no,
      enable_end_date,
      end_date,
      employee_id,
      note,
      is_deleted: 2,
      created_at: currentTime,
      created_by: req?.user?.id,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    });

    return sendResponse(res, {}, "Recurring Pay Component created successfully", 201);
  } catch (error) {
    console.error("Error While Creating Recurring Pay Component: ", error);
    return sendErrorResponse(res, error, "Error while creating Recurring Pay Component");
  }
};

// READ Recurring Pay Component(s)
module.exports.getRecurringPayComp = async (req, res) => {
  try {
    const { id } = req.query;

    let query = `
      SELECT *
      FROM ${tables.org_recurring_pay_component}
      WHERE is_deleted = 2
    `;

    if (id) query += ` AND id = ${id}`;
    query += ` ORDER BY id DESC`;

    const result = await performQuery(query);
    return sendResponse(res, result, "Recurring Pay Component(s) fetched successfully", 200);
  } catch (error) {
    console.error("Error While Fetching Recurring Pay Component(s): ", error);
    return sendErrorResponse(res, error, "Error while fetching Recurring Pay Component(s)");
  }
};

// UPDATE Recurring Pay Component
module.exports.updateRecurringPayComp = async (req, res) => {
  try {
    const { 
      id,
      value,
      currency_code,
      start_date,
      frequency,
      sequence_no,
      enable_end_date,
      end_date,
      employee_id,
      note,
    } = req.body;

    if (
      !id || !value || !currency_code || 
      !start_date || !frequency || !sequence_no || !employee_id ||
      (enable_end_date == 1 && !end_date)
    ) {
      const errorMessage = "Missing Recurring Pay Component ID";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.org_recurring_pay_component} WHERE id = ? AND is_deleted = 2`,
      [id]
    );

    if (!existingRecord || existingRecord.length === 0) {
      const errorMessage = "Recurring Pay Component not found";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    const oldRecord = existingRecord[0];

    // Duplicate check for recurring_pay_comp_id_external (excluding self)
    const duplicateCheck = await performQuery(
      `SELECT id FROM ${tables.org_recurring_pay_component} WHERE recurring_pay_comp_id_external = ? AND id <> ? AND is_deleted = 2`,
      [oldRecord.recurring_pay_comp_id_external, id]
    );

    if (duplicateCheck && duplicateCheck.length > 0) {
      const errorMessage = "Recurring Pay Component external ID already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

     // Mark existing record as deleted to maintain history
        const updateResult = await performQuery(
            `UPDATE ${tables.org_recurring_pay_component} SET ? WHERE id = ?`,
            [
                {
                    is_deleted: 1,
                    updated_at: currentTime,
                    updated_by: req?.user?.id,
                },
                id
            ]
        );

    // Update Record
    await performQuery(`INSERT INTO ${tables.org_recurring_pay_component} SET ?`, [{
      sequence_no: Number(existingRecord[0].sequence_no) + 1,
      recurring_pay_comp_id_external: oldRecord.recurring_pay_comp_id_external,
      value: value || oldRecord.value,
      currency_code: currency_code || oldRecord.currency_code,
      start_date: start_date || oldRecord.start_date,
      frequency: frequency || oldRecord.frequency,
      sequence_no: sequence_no || oldRecord.sequence_no,
      enable_end_date: enable_end_date || oldRecord.enable_end_date,
      end_date: end_date || oldRecord.end_date,
      employee_id: employee_id || oldRecord.employee_id,
      note: note || oldRecord.note,
      created_at: currentTime,
      created_by: req?.user?.id,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    }, id]);

    return sendResponse(res, {}, "Recurring Pay Component updated successfully", 200);

  } catch (error) {
    console.error("Error While Updating Recurring Pay Component: ", error);
    return sendErrorResponse(res, error, "Error while updating Recurring Pay Component");
  }
};

// DELETE Recurring Pay Component (Soft Delete)
// module.exports.deleteRecurringPayComp = async (req, res) => {
//   try {
//     const { id } = req.query;

//     if (!id) {
//       const errorMessage = "Recurring Pay Component ID is required";
//       return sendErrorResponse(res, errorMessage, errorMessage);
//     }

//     const systemTime = await getSystemTime();
//     const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

//     await performQuery(`UPDATE ${tables.org_recurring_pay_component} SET ? WHERE id = ?`, [
//       {
//         is_deleted: 1,
//         updated_at: currentTime,
//         updated_by: req?.user?.id,
//       },
//       id,
//     ]);

//     return sendResponse(res, {}, "Recurring Pay Component deleted successfully", 200);
//   } catch (error) {
//     console.error("Error While Deleting Recurring Pay Component: ", error);
//     return sendErrorResponse(res, error, "Error while deleting Recurring Pay Component");
//   }
// };
