const { sendErrorResponse, sendResponse } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");


// CREATE Pay Group
module.exports.createPayGroup = async (req, res) => {
  try {
    const {
      pay_group_id_external,
      entity_id,
      group_name,
      effective_start_date
    } = req.body;

    // Validation
    if (!pay_group_id_external || !entity_id || !group_name || !effective_start_date) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check entity_id existence in legal entity table
    const isLegalEntity = await performQuery(
      `SELECT * FROM ${tables.org_legal_entity} WHERE entity_id_external = ? AND is_deleted = 2`,
      [entity_id]
    );
    if (isLegalEntity.length === 0) {
      const errorMessage = "Legal Entity Not Found against entity_id";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check duplicate pay_group_id_external
    const duplicateCheck = await performQuery(
      `SELECT id FROM ${tables.org_pay_group} WHERE (pay_group_id_external = ? OR group_name = ?) AND is_deleted = 2`,
      [pay_group_id_external, group_name]
    );

    if (duplicateCheck && duplicateCheck.length > 0) {
      const errorMessage = "Pay Group external ID already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert record
    await performQuery(`INSERT INTO ${tables.org_pay_group} SET ?`, {
      sequence_no: 1,
      pay_group_id_external,
      entity_id,
      group_name,
      effective_start_date,
      created_at: currentTime,
      created_by: req?.user?.id,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    });

    return sendResponse(res, {}, "Pay Group created successfully", 201);
  } catch (error) {
    console.error("Error While Creating Pay Group: ", error);
    return sendErrorResponse(res, error, "Error while creating pay group");
  }
};


// READ Pay Group(s)
module.exports.getPayGroup = async (req, res) => {
  try {
    const { id } = req.query;

    let query = `
      SELECT 
        pg.*,
        le.entity_id_external, le.name, le.org_unit_code, le.org_registration_date
      FROM ${tables.org_pay_group} pg
      LEFT JOIN ${tables.org_legal_entity} le ON pg.entity_id = le.entity_id_external AND le.is_deleted = 2
      WHERE pg.is_deleted = 2
    `;

    if (id) query += ` AND pg.id = ${id}`;
    query += ` ORDER BY pg.id DESC`;

    const result = await performQuery(query);
    return sendResponse(res, result, "Pay Group(s) fetched successfully", 200);
  } catch (error) {
    console.error("Error While Fetching Pay Group(s): ", error);
    return sendErrorResponse(res, error, "Error while fetching pay group(s)");
  }
};


// UPDATE Pay Group
module.exports.updatePayGroup = async (req, res) => {
  try {
    const { 
      id,
      entity_id,
      group_name,
      effective_start_date,
      status 
    } = req.body;

    // Validation
    if (!id || !entity_id || !group_name || !effective_start_date) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check existence of the record to be updated
    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.org_pay_group} WHERE id = ? AND is_deleted = 2`,
      [id]
    );

    if (!existingRecord || existingRecord.length === 0) {
      const errorMessage = "Pay Group not found";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    const oldRecord = existingRecord[0];

    // Duplicate check for pay_group_id_external (excluding self)
    const duplicateCheck = await performQuery(
      `SELECT id FROM ${tables.org_pay_group} WHERE (pay_group_id_external = ? OR group_name = ?) AND id <> ? AND is_deleted = 2`,
      [oldRecord.pay_group_id_external, group_name, id]
    );

    if (duplicateCheck && duplicateCheck.length > 0) {
      const errorMessage = "Pay Group external ID already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Mark existing record as deleted to maintain history
        const updateResult = await performQuery(
            `UPDATE ${tables.org_pay_group} SET ? WHERE id = ?`,
            [
                {
                    is_deleted: 1,
                    updated_at: currentTime,
                    updated_by: req?.user?.id,
                },
                id
            ]
        );
      
    // Update Record
    await performQuery(`INSERT INTO ${tables.org_pay_group} SET ?`, [{
        sequence_no: Number(existingRecord[0].sequence_no) + 1,
        pay_group_id_external: oldRecord.pay_group_id_external,
        entity_id: entity_id || oldRecord.entity_id,
        group_name: group_name || oldRecord.group_name,
        effective_start_date: effective_start_date || oldRecord.effective_start_date,
        status: status || oldRecord.status,
        is_deleted: 2,
        created_at: currentTime,
        created_by: req?.user?.id,
        updated_at: currentTime,
        updated_by: req?.user?.id,
    }, id]);

    return sendResponse(res, {}, "Pay Group updated successfully", 200);
  } catch (error) {
    console.error("Error While Updating Pay Group: ", error);
    return sendErrorResponse(res, error, "Error while updating pay group");
  }
};


// DELETE Pay Group (Soft Delete)
// module.exports.deletePayGroup = async (req, res) => {
//   try {
//     const { id } = req.query;

//     if (!id) {
//       const errorMessage = "Pay Group ID is required";
//       return sendErrorResponse(res, errorMessage, errorMessage);
//     }

//     const systemTime = await getSystemTime();
//     const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

//     await performQuery(`UPDATE ${tables.org_pay_group} SET ? WHERE id = ?`, [
//       {
//         is_deleted: 1,
//         updated_at: currentTime,
//         updated_by: req?.user?.id,
//       },
//       id,
//     ]);

//     return sendResponse(res, {}, "Pay Group deleted successfully", 200);
//   } catch (error) {
//     console.error("Error While Deleting Pay Group: ", error);
//     return sendErrorResponse(res, error, "Error while deleting pay group");
//   }
// };
