const { sendErrorResponse, sendResponse } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");
const { isUndefined } = require("lodash");


// CREATE Location Group
module.exports.createLocationGroup = async (req, res) => {
  try {
    const {
      location_group_id_external,
      location_group_name,
      effective_start_date
    } = req.body;

    // Required field validation
    if (!location_group_id_external || !location_group_name || !effective_start_date) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check for duplicate location_group_id_external or name
    const existingRecord = await performQuery(
      `SELECT id FROM ${tables.org_location_group} 
       WHERE (location_group_id_external = ? OR location_group_name = ?) AND is_deleted = 2`,
      [location_group_id_external, location_group_name]
    );
    if (existingRecord.length > 0) {
      const errorMessage = "Location group with this ID or name already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Get system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert into DB
    const newRecord = await performQuery(
      `INSERT INTO ${tables.org_location_group} SET ?`, {
        location_group_id_external,
        location_group_name,
        effective_start_date,
        created_at: currentTime,
        created_by: req?.user?.id,
        updated_at: currentTime,
        updated_by: req?.user?.id
      }
    );

    return sendResponse(res, {}, "Location group created successfully", 201);
  } catch (error) {
    return sendErrorResponse(res, error, "Error while creating location group");
  }
};


// READ Location Groups
module.exports.getLocationGroup = async (req, res) => {
  try {
    const { id, location_group_id_external } = req.query;

    let query = `
      SELECT 
        lg.*,
        cre.display_name AS created_by_name,
        upd.display_name AS updated_by_name
      FROM ${tables.org_location_group} AS lg
      LEFT JOIN ${tables.per_person} AS cre ON lg.created_by = cre.id
      LEFT JOIN ${tables.per_person} AS upd ON lg.updated_by = upd.id
      WHERE lg.is_deleted = 2
    `;

    if (id) query += ` AND lg.id = ${id}`;
    if (location_group_id_external) query += ` AND lg.location_group_id_external = '${location_group_id_external}'`;

    query += ` ORDER BY lg.id DESC`;

    const locationGroups = await performQuery(query);

    return sendResponse(res, locationGroups, "Location group(s) fetched successfully", 200);
  } catch (error) {
    return sendErrorResponse(res, error, "Error while fetching location group(s)");
  }
};


// UPDATE Location Group
module.exports.updateLocationGroup = async (req, res) => {
  try {
    const {
      id,
      location_group_name,
      effective_start_date,
      status
    } = req.body;

    // Required field validation
    if (!id || !location_group_name || !effective_start_date) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check if record exists
    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.org_location_group} WHERE id = ? AND is_deleted = 2`,
      [id]
    );
    if (!existingRecord || existingRecord.length === 0) {
      const errorMessage = "Location group not found";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    const previousRecord = existingRecord[0];

    // Check duplicate name/ID
    const duplicate = await performQuery(
      `SELECT id FROM ${tables.org_location_group} 
       WHERE (location_group_id_external = ? OR location_group_name = ?) AND id <> ? AND is_deleted = 2`,
      [previousRecord.location_group_id_external, location_group_name, id]
    );
    if (duplicate.length > 0) {
      const errorMessage = "Duplicate location group ID or name";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Get system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");
    
    // Update Record to maintain history
    const updatedRecord = await performQuery(
      `UPDATE ${tables.org_location_group} SET ? Where id = ?`, [{
        is_deleted: 2,
        updated_at: currentTime,
        updated_by: req?.user?.id
      }, id]
    );

    // Insert new record with updated data
    await performQuery(
      `INSERT INTO ${tables.org_location_group} SET ?`, {
        location_group_id_external: previousRecord.location_group_id_external,
        sequence_no: Number(previousRecord.sequence_no) + 1,
        location_group_name: location_group_name || previousRecord.location_group_name,
        effective_start_date: effective_start_date || previousRecord.effective_start_date,
        status: status || previousRecord.status,
        created_at: currentTime,
        created_by: req?.user?.id,
        updated_at: currentTime,
        updated_by: req?.user?.id
      }
    );


    return sendResponse(res, {}, "Location group updated successfully", 200);
  } catch (error) {
    return sendErrorResponse(res, error, "Error while updating location group");
  }
};


// DELETE (Soft Delete)
// module.exports.deleteLocationGroup = async (req, res) => {
//   try {
//     const { id } = req.query;

//     if (isUndefined(id)) {
//       const errorMessage = "Location group ID is required";
//       return sendErrorResponse(res, errorMessage, errorMessage);
//     }

//     // Get system time
//     const systemTime = await getSystemTime();
//     const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

//     // Soft delete
//     await performQuery(
//       `UPDATE ${tables.org_location_group} SET ? WHERE id = ?`,
//       [{ is_deleted: 1, updated_at: currentTime, updated_by: req?.user?.id }, id]
//     );

//     return sendResponse(res, {}, "Location group deleted successfully", 200);
//   } catch (error) {
//     return sendErrorResponse(res, error, "Error while deleting location group");
//   }
// };
