const { sendErrorResponse, sendResponse } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");

// CREATE Cost Center
module.exports.createCostCenter = async (req, res) => {
  try {
    const {
      cost_center_id_external,
      name_us,
      name_sa,
      description_us,
      description_sa,
      gl_statement_code,
      parent_cost_center_id,
      cost_center_manager_id,
      entity_id,
      effective_start_date,
    } = req.body;

    // Validation
    if (!cost_center_id_external || !name_us || !effective_start_date || !entity_id) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check entity_id if exisit in legal entity tabel
    const isLegalEntity = await performQuery(
      `SELECT * FROM ${tables.org_legal_entity} WHERE entity_id_external = ? AND is_deleted = 2`,
      [entity_id]
    );
    if (isLegalEntity.length === 0) {
      const errorMessage = "Legal Entity Not Found against entity_id";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    if(parent_cost_center_id){
      // Check parent_cost_center_id if exisit in cost center tabel
      const isParentCostCenter = await performQuery(
        `SELECT * FROM ${tables.org_cost_center} WHERE cost_center_id_external = ? AND is_deleted = 2`,
        [parent_cost_center_id]
      );
      if (isParentCostCenter.length === 0) {
        const errorMessage = "Parent Cost Center Not Found against parent_cost_center_id";
        return sendErrorResponse(res, errorMessage, errorMessage);
      }
    }

    if(cost_center_manager_id){
      // Check cost_center_manager_id if exisit in employee tabel
      const isCostCenterManager = await performQuery(
        `SELECT * FROM ${tables.emp_employee} WHERE employee_id_external = ? AND is_deleted = 2`,
        [cost_center_manager_id]
      );
      if (isCostCenterManager.length === 0) {
        const errorMessage = "Cost Center Manager Not Found against cost_center_manager_id";
        return sendErrorResponse(res, errorMessage, errorMessage);
      }
    }

    // Check for duplicate cost_center_id_external
    const duplicateCheck = await performQuery(
      `SELECT id FROM ${tables.org_cost_center} WHERE cost_center_id_external = ? AND is_deleted = 2`,
      [cost_center_id_external]
    );

    if (duplicateCheck && duplicateCheck.length > 0) {
      const errorMessage = "Cost center external ID already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert record
    await performQuery(`INSERT INTO ${tables.org_cost_center} SET ?`, {
      cost_center_id_external,
      name_us,
      name_sa,
      description_us,
      description_sa,
      gl_statement_code,
      parent_cost_center_id: parent_cost_center_id,
      cost_center_manager_id: cost_center_manager_id,
      entity_id: entity_id,
      effective_start_date,
      created_at: currentTime,
      created_by: req?.user?.id,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    });

    return sendResponse(res, {}, "Cost Center created successfully", 201);
  } catch (error) {
    console.error("Error While Creating Cost Center: ", error);
    return sendErrorResponse(res, error, "Error while creating cost center");
  }
};

// READ Cost Center(s)
module.exports.getCostCenter = async (req, res) => {
  try {
    const { id, entity_id, entity_id_external} = req.query;

    let query = `
      SELECT 
        cc.*, p.display_name AS cost_center_manager_name,
        parent_cc.cost_center_id_external AS parent_cost_center_id_external, parent_cc.name_us AS parent_cost_center_name_us,
        le.name AS legal_entity_name, e.employee_id_external AS cost_center_manager_id_external, p.display_name AS cost_center_manager_name
      FROM ${tables.org_cost_center} cc
      LEFT JOIN ${tables.org_legal_entity} le ON cc.entity_id = le.entity_id_external AND le.is_deleted = 2
      LEFT JOIN ${tables.emp_employee} e ON cc.cost_center_manager_id = e.employee_id_external AND e.is_deleted = 2
      LEFT JOIN ${tables.per_person} p ON e.person_id_external = p.person_id_external AND p.is_deleted = 2
      LEFT JOIN ${tables.org_cost_center} parent_cc ON cc.parent_cost_center_id = parent_cc.cost_center_id_external AND parent_cc.is_deleted = 2
      WHERE cc.is_deleted = 2
    `;

    if (id) query += ` AND cc.id = ${id}`;
    if (entity_id_external) query += ` AND le.entity_id_external = '${entity_id_external}'`;
    query += ` ORDER BY cc.id DESC`;

    const result = await performQuery(query);
    return sendResponse(res, result, "Cost Center(s) fetched successfully", 200);

  } catch (error) {
    console.error("Error While Fetching Cost Center(s): ", error);
    return sendErrorResponse(res, error, "Error while fetching cost center(s)");
  }
};

// UPDATE Cost Center
module.exports.updateCostCenter = async (req, res) => {
  try {
    const { 
      id,
      name_us,
      name_sa,
      description_us,
      description_sa,
      gl_statement_code,
      parent_cost_center_id,
      cost_center_manager_id,
      entity_id,
      effective_start_date,
      status
    } = req.body;

    // Validation
    if (!id || !name_us || !effective_start_date || !entity_id) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.org_cost_center} WHERE id = ? AND is_deleted = 2`,
      [id]
    );
    if (!existingRecord || existingRecord.length === 0) {
      const errorMessage = "Cost Center not found";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    const oldRecord = existingRecord[0];

        // Check entity_id if exisit in legal entity tabel
    const isLegalEntity = await performQuery(
      `SELECT * FROM ${tables.org_legal_entity} WHERE entity_id_external = ? AND is_deleted = 2`,
      [entity_id]
    );
    if (isLegalEntity.length === 0) {
      const errorMessage = "Legal Entity Not Found against entity_id";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    if(parent_cost_center_id){
      // Check parent_cost_center_id if exisit in cost center tabel
      const isParentCostCenter = await performQuery(
        `SELECT * FROM ${tables.org_cost_center} WHERE cost_center_id_external = ? AND is_deleted = 2`,
        [parent_cost_center_id]
      );
      if (isParentCostCenter.length === 0) {
        const errorMessage = "Parent Cost Center Not Found against parent_cost_center_id";
        return sendErrorResponse(res, errorMessage, errorMessage);
      }
    }

    if(cost_center_manager_id){
      // Check cost_center_manager_id if exisit in employee tabel
      const isCostCenterManager = await performQuery(
        `SELECT * FROM ${tables.emp_employee} WHERE employee_id_external = ? AND is_deleted = 2`,
        [cost_center_manager_id]
      );
      if (isCostCenterManager.length === 0) {
        const errorMessage = "Cost Center Manager Not Found against cost_center_manager_id";
        return sendErrorResponse(res, errorMessage, errorMessage);
      }
    }

    // Duplicate check for cost_center_id_external (excluding self)
    const duplicateCheck = await performQuery(
      `SELECT id FROM ${tables.org_cost_center} WHERE cost_center_id_external = ? AND id <> ? AND is_deleted = 2`,
      [oldRecord.cost_center_id_external, id]
    );

    if (duplicateCheck && duplicateCheck.length > 0) {
      const errorMessage = "Cost center external ID already exists";
      return sendErrorResponse(res, errorMessage, errorMessage)
    }

    // System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Update record to maintain history
    await performQuery(`UPDATE ${tables.org_cost_center} SET ? Where id = ?`, [{
      is_deleted: 1,
      updated_at: currentTime,
      updated_by: req.user?.id,
    }, id]);

    // Insert new record with updated data
    await performQuery(`INSERT INTO ${tables.org_cost_center} SET ?`, {
      cost_center_id_external: oldRecord.cost_center_id_external,
      sequence_no: Number(oldRecord.sequence_no) + 1,
      name_us: name_us || oldRecord.name_us,
      name_sa: name_sa || oldRecord.name_sa,
      description_us: description_us || oldRecord.description_us,
      description_sa: description_sa || oldRecord.description_sa,
      gl_statement_code: gl_statement_code || oldRecord.gl_statement_code,
      parent_cost_center_id: parent_cost_center_id || oldRecord.parent_cost_center_id,
      cost_center_manager_id: cost_center_manager_id || oldRecord.cost_center_manager_id,
      entity_id: entity_id || oldRecord.entity_id,
      effective_start_date: effective_start_date || oldRecord.effective_start_date,
      status: status || oldRecord.status,
      created_at: currentTime,
      created_by: req?.user?.id,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    });

    return sendResponse(res, {}, "Cost Center updated successfully", 200);
    
  } catch (error) {
    console.error("Error While Updating Cost Center: ", error);
    return sendErrorResponse(res, error, "Error while updating cost center");
  }
};

// DELETE Cost Center (Soft Delete)
// module.exports.deleteCostCenter = async (req, res) => {
//   try {
//     const { id } = req.query;

//     if (!id) {
//       const errorMessage = "Cost Center ID is required";
//       return sendErrorResponse(res, errorMessage, errorMessage);
//     }

//     const systemTime = await getSystemTime();
//     const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

//     await performQuery(`UPDATE ${tables.org_cost_center} SET ? WHERE id = ?`, [
//       {
//         is_deleted: 1,
//         updated_at: currentTime,
//         updated_by: req?.user?.id,
//       },
//       id,
//     ]);

//     return sendResponse(res, {}, "Cost Center deleted successfully", 200);
//   } catch (error) {
//     console.error("Error While Deleting Cost Center: ", error);
//     return sendErrorResponse(res, error, "Error while deleting cost center");
//   }
// };
