const { sendErrorResponse, sendResponse } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");

// CREATE Compensation
module.exports.createCompensation = async (req, res) => {
  try {
    const {
      compansation_id_external,
      employee_id_external,
      event_date,
      event_reason,
      end_date,
      pay_group_id,
      recurring_pay_component_id,
      non_recurring_pay_component_id,
    } = req.body;

    // Validation
    if (
      !compansation_id_external ||
      !employee_id_external ||
      !event_date ||
      !pay_group_id ||
      !event_reason ||
      (!recurring_pay_component_id && !non_recurring_pay_component_id)
    ) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Validate pay group existence
    const payGroupExists = await performQuery(
      `SELECT * FROM ${tables.org_pay_group} WHERE id = ? AND is_deleted = 2`,
      [pay_group_id]
    );
    if (payGroupExists.length === 0) {
      const errorMessage = "Employee not found against employee_id";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // validate employee existence
    const employeeExists = await performQuery(
      `SELECT * FROM ${tables.emp_employee} WHERE employee_id_external = ? AND is_deleted = 2`,
      [employee_id_external]
    );
    if (employeeExists.length === 0) {
      const errorMessage = "Employee not found against employee_id_external";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Validate recurring pay component existence
    if (recurring_pay_component_id) {
      const recurringPayCompoExists = await performQuery(
        `SELECT * FROM ${tables.org_recurring_pay_component} WHERE id = ? AND is_deleted = 2`,
        [recurring_pay_component_id]
      );
      if (recurringPayCompoExists.length === 0) {
        const errorMessage = "Employee not found against employee_id";
        return sendErrorResponse(res, errorMessage, errorMessage);
      }
    }
    if (non_recurring_pay_component_id) {
      // Validate non recurring pay component existence
      const nonRecurringPayCompoExists = await performQuery(
        `SELECT * FROM ${tables.org_non_recurring_pay_component} WHERE id = ? AND is_deleted = 2`,
        [non_recurring_pay_component_id]
      );

      if (nonRecurringPayCompoExists.length === 0) {
        const errorMessage = "Employee not found against employee_id";
        return sendErrorResponse(res, errorMessage, errorMessage);
      }
    }

    // Duplicate Check
    const duplicateCheck = await performQuery(
      `SELECT id FROM ${tables.org_compansation} WHERE compansation_id_external = ? AND is_deleted = 2`,
      [compansation_id_external]
    );
    if (duplicateCheck && duplicateCheck.length > 0) {
      const errorMessage = "Compensation external ID already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert record
    await performQuery(`INSERT INTO ${tables.org_compansation} SET ?`, {
      compansation_id_external,
      employee_id_external,
      event_date,
      event_reason,
      end_date,
      pay_group_id,
      recurring_pay_component_id,
      non_recurring_pay_component_id,
      is_deleted: 2,
      created_at: currentTime,
      created_by: req?.user?.id,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    });

    return sendResponse(res, {}, "Compensation created successfully", 201);
  } catch (error) {
    console.error("Error While Creating Compensation: ", error);
    return sendErrorResponse(res, error, "Error while creating Compensation");
  }
};

// READ Compensation(s)
module.exports.getCompensation = async (req, res) => {
  try {
    const { id } = req.query;

    let query = `
      SELECT *
      FROM ${tables.org_compansation}
      WHERE is_deleted = 2
    `;

    if (id) query += ` AND id = ${id}`;
    query += ` ORDER BY id DESC`;

    const result = await performQuery(query);
    return sendResponse(
      res,
      result,
      "Compensation record(s) fetched successfully",
      200
    );
  } catch (error) {
    console.error("Error While Fetching Compensation(s): ", error);
    return sendErrorResponse(
      res,
      error,
      "Error while fetching Compensation(s)"
    );
  }
};

// UPDATE Compensation (Versioned Update)
module.exports.updateCompensation = async (req, res) => {
  try {
    const {
      id,
      compansation_id_external,
      employee_id_external,
      event_date,
      event_reason,
      end_date,
      pay_group_id,
      recurring_pay_component_id,
      non_recurring_pay_component_id,
    } = req.body;

    if (
      !id ||
      !compansation_id_external ||
      !employee_id_external ||
      !event_date ||
      !pay_group_id ||
      !event_reason ||
      (!recurring_pay_component_id && !non_recurring_pay_component_id)
    ) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.org_compansation} WHERE id = ? AND is_deleted = 2`,
      [id]
    );

    if (!existingRecord || existingRecord.length === 0) {
      const errorMessage = "Compensation record not found";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    const oldRecord = existingRecord[0];

    // Duplicate check (excluding current)
    const duplicateCheck = await performQuery(
      `SELECT id FROM ${tables.org_compansation} WHERE compansation_id_external = ? AND id <> ? AND is_deleted = 2`,
      [compansation_id_external, id]
    );

    if (duplicateCheck && duplicateCheck.length > 0) {
      const errorMessage = "Compensation external ID already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // check is pay_group, recurring pay component, non recurring pay component exist
    const [row] = await performQuery(
      `
        SELECT
          EXISTS (
            SELECT 1 FROM ${tables.org_pay_group}
            WHERE id = ? AND is_deleted = 2
          ) AS pay_group_ok,

          ( ? IS NULL OR EXISTS (
            SELECT 1 FROM ${tables.org_recurring_pay_component}
            WHERE id = ? AND is_deleted = 2
          )) AS recurring_ok,

          ( ? IS NULL OR EXISTS (
            SELECT 1 FROM ${tables.org_non_recurring_pay_component}
            WHERE id = ? AND is_deleted = 2
          )) AS non_recurring_ok,

          ( ? IS NULL OR EXISTS (
            SELECT 1 FROM ${tables.emp_employee}
            WHERE employee_id_external = ? AND is_deleted = 2
          )) AS employee_ok

      `,
      [
        pay_group_id,
        recurring_pay_component_id,
        recurring_pay_component_id,
        non_recurring_pay_component_id,
        non_recurring_pay_component_id,
        employee_id_external,
        employee_id_external,
      ]
    );

    if (!row.pay_group_ok) {
      const msg = "Pay group not found (id / is_deleted mismatch)";
      return sendErrorResponse(res, msg, msg);
    }
    if (!row.recurring_ok && recurring_pay_component_id) {
      const msg =
        "Recurring pay component not found (id / is_deleted mismatch)";
      return sendErrorResponse(res, msg, msg);
    }
    if (!row.non_recurring_ok && non_recurring_pay_component_id) {
      const msg =
        "Non-recurring pay component not found (id / is_deleted mismatch)";
      return sendErrorResponse(res, msg, msg);
    }

    // System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Update Record record
    await performQuery(`UPDATE ${tables.org_compansation} SET ? Where id = ?`, [
      {
        employee_id_external:
          employee_id_external || oldRecord.employee_id_external,
        event_date: event_date || oldRecord.event_date,
        event_reason: event_reason || oldRecord.event_reason,
        end_date: end_date || oldRecord.end_date,
        pay_group_id: pay_group_id || oldRecord.pay_group_id,
        recurring_pay_component_id:
          recurring_pay_component_id || oldRecord.recurring_pay_component_id,
        non_recurring_pay_component_id:
          non_recurring_pay_component_id ||
          oldRecord.non_recurring_pay_component_id,
        updated_at: currentTime,
        updated_by: req?.user?.id,
      },
      id,
    ]);

    return sendResponse(res, {}, "Compensation updated successfully", 200);
  } catch (error) {
    console.error("Error While Updating Compensation: ", error);
    return sendErrorResponse(res, error, "Error while updating Compensation");
  }
};

// DELETE Compensation (Soft Delete)
// module.exports.deleteCompensation = async (req, res) => {
//   try {
//     const { id } = req.query;

//     if (!id) {
//       const errorMessage = "Compensation ID is required";
//       return sendErrorResponse(res, errorMessage, errorMessage);
//     }

//     const systemTime = await getSystemTime();
//     const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

//     await performQuery(`UPDATE ${tables.org_compansation} SET ? WHERE id = ?`, [
//       {
//         is_deleted: 1,
//         updated_at: currentTime,
//         updated_by: req?.user?.id,
//       },
//       id,
//     ]);

//     return sendResponse(res, {}, "Compensation deleted successfully", 200);
//   } catch (error) {
//     console.error("Error While Deleting Compensation: ", error);
//     return sendErrorResponse(res, error, "Error while deleting Compensation");
//   }
// };
