const { sendErrorResponse, sendResponse } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");

// CREATE Business Unit
module.exports.createBusinessUnit = async (req, res) => {
  try {
    const {
      bu_id_external,
      entity_id,
      effective_start_date,
      description,
      description_arabic,
    } = req.body;

    // Validation
    if (!bu_id_external || !entity_id || !effective_start_date) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check entity_id if exists in legal entity table
    const isLegalEntity = await performQuery(
      `SELECT * FROM ${tables.org_legal_entity} WHERE entity_id_external = ? AND is_deleted = 2`,
      [entity_id]
    );
    if (isLegalEntity.length === 0) {
      const errorMessage = "Legal Entity Not Found against entity_id";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check duplicate bu_id_external 
    const existingRecord = await performQuery(
      `SELECT id FROM ${tables.org_business_unit} WHERE bu_id_external = ? AND is_deleted = 2`,
      [bu_id_external]
    );
    if (existingRecord && existingRecord.length > 0) {
      const errorMessage = "Entity already exists with this bu_id_external";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert
    await performQuery(`INSERT INTO ${tables.org_business_unit} SET ?`, {
      sequence_no: 1,
      bu_id_external,
      entity_id,
      effective_start_date,
      description,
      description_arabic,
      created_at: currentTime,
      created_by: req?.user?.id,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    });

    // Success Response
    return sendResponse(res, {}, "Entity created successfully", 201);

  } catch (error) {
    console.error("Error while creating Entity:", error);
    return sendErrorResponse(res, error, "Error while creating entity");
  }
};

// READ Business Unit
module.exports.getBusinessUnit = async (req, res) => {
  try {
    const { 
      id, 
      bu_id_external,
      entity_id,
    } = req.query;

    // Base Query
    let query = `
      SELECT 
        bu.*,
        le.entity_id_external, le.name AS legal_entity_name
      FROM ${tables.org_business_unit} AS bu
      LEFT JOIN ${tables.org_legal_entity} AS le ON bu.entity_id = le.entity_id_external AND le.is_deleted = 2
      WHERE bu.is_deleted = 2`;

    // Apply filters
    if (id) query += ` AND bu.id = ${id}`;
    if (bu_id_external) query += ` AND bu.bu_id_external = '${bu_id_external}'`;
    if (entity_id) query += ` AND bu.entity_id_external = '${entity_id}'`;

    // Order by latest
    query += ` ORDER BY bu.id DESC`;

    // Execute Query
    const entities = await performQuery(query);
    
    // Success Response
    return sendResponse(res, entities, "Entity record(s) fetched successfully", 200);
    
  } catch (error) {
    return sendErrorResponse(res, error, "Error while fetching entities");
  }
};

// UPDATE business Unit
module.exports.updateBusinessUnit = async (req, res) => {
  try {
    const {
      id,
      entity_id,
      effective_start_date,
      description,
      description_arabic,
      status,
    } = req.body;

    if (!id || !entity_id || !effective_start_date) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check if entity exists
    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.org_business_unit} WHERE id = ? AND is_deleted = 2`,
      [id]
    );
    console.log("existingRecord: ", existingRecord);
    
    if (!existingRecord || existingRecord.length === 0) {
      const errorMessage = "Record not found";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    const oldRecord = existingRecord[0];

    // Check for duplicate bu_id_external
    const duplicateBUCheck = await performQuery(
      `SELECT id FROM ${tables.org_business_unit} WHERE bu_id_external = ? AND id <> ? AND is_deleted = 2`,
      [oldRecord.bu_id_external, id]
    );
    if (duplicateBUCheck && duplicateBUCheck.length > 0) {
      const errorMessage = "Business ID External already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Mark existing record as deleted to maintain history
        const updateResult = await performQuery(
            `UPDATE ${tables.org_business_unit} SET ? WHERE id = ?`,
            [
                {
                    is_deleted: 1,
                    updated_at: currentTime,
                    updated_by: req?.user?.id,
                },
                id
            ]
        );

    // UPDATE record
    await performQuery(`INSERT INTO ${tables.org_business_unit} SET ?`, [
      {
        bu_id_external: oldRecord.bu_id_external,
        sequence_no: Number(existingRecord[0].sequence_no) +1,
        entity_id: entity_id || oldRecord.entity_id,
        effective_start_date: effective_start_date || oldRecord.effective_start_date,
        description: description || oldRecord.description,
        description_arabic: description_arabic || oldRecord.description_arabic,
        status: status || oldRecord.status,
        created_at: currentTime,
        created_by: req?.user?.id,
        updated_at: currentTime,
        updated_by: req?.user?.id,
      }, id]
    );

    // Success Response
    return sendResponse(res, {}, "Entity updated successfully", 200);

  } catch (error) {
    console.log("Error while updating entity: ", error);
    return sendErrorResponse(res, error, "Error while updating entity");
  }
};

// DELETE Business Unit
module.exports.deleteBusinessUnit = async (req, res) => {
  try {
    const { id } = req.query;

    if (!id) {
      const errorMessage = "Entity ID is required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Soft delete
    await performQuery(
      `UPDATE ${tables.org_business_unit} SET ? WHERE id = ?`,
      [
        {
          is_deleted: 1,
          updated_at: currentTime,
          updated_by: req?.user?.id,
        },
        id,
      ]
    );

    return sendResponse(res, {}, "Entity deleted successfully", 200);
  } catch (error) {
    return sendErrorResponse(res, error, "Error while deleting entity");
  }
};

