const { sendErrorResponse, sendResponse, } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");

// --> Create Holiday Assignments
module.exports.createWorkSchedule = async (req, res) => {
    try {
        // Extract required fields from request body
        const {
            schedule_id_external,
            external_name,
            starting_date,
            country,
            model,
            cross_midnight_allowed,
            user,
            average_hours_per_day,
            average_hours_per_week,
            average_hours_per_month,
            average_hours_per_year,
            average_working_days_per_week,
            shift_classification,
            time_recording_variant,
        } = req.body;

        // Validate required fields
        if (
            !schedule_id_external || !external_name || !starting_date || !country || !model ||
            cross_midnight_allowed === undefined || time_recording_variant === undefined
        ) {
            return sendErrorResponse(
                res,
                "Schedule ID, External Name, Starting Date, Country, and Model are required",
                "Schedule ID, External Name, Starting Date, Country, and Model are required"
            );
        }

        // Check for existing active work schedule with the same external ID
        const existingSchedule = await performQuery(
            `SELECT * FROM ${tables.work_schedule} WHERE schedule_id_external = ? AND is_active = 1`,
            [schedule_id_external]
        );
        if (existingSchedule.length > 0) {
            return sendErrorResponse(res, "Work Schedule with this External ID already exists", "Work Schedule with this External ID already exists");
        }

        // Check if same external name exists for the country
        const existingNameForCountry = await performQuery(
            `SELECT * FROM ${tables.work_schedule} WHERE external_name = ? AND country = ? AND is_active = 1`,
            [external_name, country]
        );
        if (existingNameForCountry.length > 0) {
            return sendErrorResponse(res, "Work Schedule with this External Name already exists for the specified country", "Work Schedule with this External Name already exists for the specified country");
        }

        // Check is model exist in picklist
        const checkModel = await performQuery(
            `SELECT * FROM ${tables.picklist_master} WHERE (picklist_id = 18 and id = ?) AND is_deleted = 2`,
            [model]
        );
        if (checkModel.length === 0) {
            return sendErrorResponse(res, "Invalid model value", "Invalid model value");
        }

        // Check is time_recording_variant exist in picklist
        const checkTimeRecordingVarient = await performQuery(
            `SELECT * FROM ${tables.picklist_master} WHERE (picklist_id = 19 and id = ?) AND is_deleted = 2`,
            [time_recording_variant]
        );
        if (checkTimeRecordingVarient.length === 0) {
            return sendErrorResponse(res, "Invalid time recording varient value", "Invalid time recording varient value");
        }

        // Get current system time
        const systemTime = await getSystemTime();
        const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

        // Insert new work schedule record
        const result = await performQuery(
            `INSERT INTO ${tables.work_schedule} SET ?`, {
                sequence_no: 1,
                schedule_id_external,
                external_name,
                starting_date,
                country,
                model,
                cross_midnight_allowed,
                user,
                average_hours_per_day: average_hours_per_day || null,
                average_hours_per_week: average_hours_per_week || null,
                average_hours_per_month: average_hours_per_month || null,
                average_hours_per_year: average_hours_per_year || null,
                average_working_days_per_week: average_working_days_per_week || null,
                shift_classification: shift_classification || null,
                time_recording_variant,
                created_by: req?.user?.id,
                created_at: currentTime,
                updated_by: req?.user?.id,
                updated_at: currentTime,
            }
        );

        // Send success response
        return sendResponse(res, {}, "Work Schedule Created Successfully", 201);

    } catch (error) {
        console.error("Error While Creating Work Schedule: ", error);
        return sendErrorResponse(res, "Internal Server Error", "Internal Server Error");
    }
}


module.exports.getWorkSchedules = async (req, res) => {
    try {
        // Extract query parameters
        const { id, schedule_id_external } = req.query;

        // Build query dynamically based on provided parameters
        let query = `
            SELECT 
                ws.id, ws.schedule_id_external, 
                ws.external_name, ws.starting_date, 
                ws.country, 
                ws.model, pm_model.picklist_option AS model_value, 
                ws.cross_midnight_allowed, 
                ws.user, 
                ws.average_hours_per_day, ws.average_hours_per_week, 
                ws.average_hours_per_month, ws.average_hours_per_year, 
                ws.average_working_days_per_week, ws.shift_classification, 
                ws.time_recording_variant, pm_time_recording.picklist_option AS time_recording_variant_value,
                ws.sequence_no,
                ws.created_by, ws.created_at, 
                ws.updated_by, ws.updated_at,
                wsd.id AS work_schedule_detail_id, wsd.day,
                wsd.day_model_id_external,
                wsd.day_model_category, pm_day_category.picklist_option AS day_model_category_value,
                wsd.created_by, wsd.created_at, 
                wsd.updated_by, wsd.updated_at,
                wsd.sequence_no,
                dm.id AS segment_id, dm.day_model_id_external, dm.external_name AS day_model_name,
                dms.id AS day_model_segment_id, dms.start_time, dms.end_time, dms.duration, dms.day_model_segments_category, pm_day_model_segment_category.picklist_option AS day_model_segment_category_value
            FROM ${tables.work_schedule} ws 
            LEFT JOIN ${tables.picklist_master} pm_model ON ws.model = pm_model.id AND pm_model.picklist_id = 18 AND pm_model.is_deleted = 2
            LEFT JOIN ${tables.picklist_master} pm_time_recording ON ws.time_recording_variant = pm_time_recording.id AND pm_time_recording.picklist_id = 19 AND pm_time_recording.is_deleted = 2
            LEFT JOIN ${tables.work_schedule_details} wsd ON ws.schedule_id_external = wsd.schedule_id_external AND wsd.is_active = 1
            LEFT JOIN ${tables.picklist_master} pm_day_category ON wsd.day_model_category = pm_day_category.id AND pm_day_category.picklist_id = 21 AND pm_day_category.is_deleted = 2
            LEFT JOIN ${tables.day_model} dm ON wsd.day_model_id_external = dm.day_model_id_external AND dm.is_active = 1
            LEFT JOIN ${tables.day_model_segments} dms ON dm.day_model_id_external = dms.day_model_id_external AND dms.is_active = 1
            LEFT JOIN ${tables.picklist_master} pm_day_model_segment_category ON dms.day_model_segments_category = pm_day_model_segment_category.id AND pm_day_model_segment_category.picklist_id = 20 AND pm_day_model_segment_category.is_deleted = 2
            WHERE ws.is_active = 1`;

        if (id) query += ` AND ws.id = ${id}`;
        if (schedule_id_external) query += ` AND ws.schedule_id_external = '${schedule_id_external}'`;
        query += ` ORDER BY ws.id DESC`;    

        const workSchedules = await performQuery(query);
        const transformedWorkSchedules = await TransformData(workSchedules);
        // return sendResponse(res, workSchedules, "Work Schedules Fetched Successfully", 200);
        return sendResponse(res, transformedWorkSchedules, "Work Schedules Fetched Successfully", 200);

    } catch (error) {
        console.log("Error While Fetching Work Schedules: ", error);
        return sendErrorResponse(res, "Internal Server Error", "Internal Server Error");       
    }
}

module.exports.updateWorkSchedule = async (req, res) => {
    try {
        // Extract required fields from request body
        const {
            id,
            external_name,
            starting_date,
            country,
            model,
            cross_midnight_allowed,
            user,
            average_hours_per_day,
            average_hours_per_week,
            average_hours_per_month,
            average_hours_per_year,
            average_working_days_per_week,
            shift_classification,
            time_recording_variant,
        } = req.body;

        // Validate required fields
        if (
            !id || !external_name || !starting_date || !country || !model ||
            cross_midnight_allowed === undefined || time_recording_variant === undefined
        ) {
            return sendErrorResponse(
                res,
                "Schedule ID, External Name, Starting Date, Country, and Model are required",
                "Schedule ID, External Name, Starting Date, Country, and Model are required"
            );
        }

        // Check is Record exist against the id
        const existingRecord = await performQuery(
            `SELECT * FROM ${tables.work_schedule} WHERE id = ? AND is_active = 1`,
            [id]
        );
        if (existingRecord.length === 0) {
            return sendErrorResponse(res, "No active Work Schedule found with the provided ID", "No active Work Schedule found with the provided ID");
        }

        // Check if same external name exists for the country excluding current record
        const existingNameForCountry = await performQuery(
            `SELECT * FROM ${tables.work_schedule} WHERE external_name = ? AND country = ? AND id <> ? AND is_active = 1`,
            [external_name, country, id]
        );
        if (existingNameForCountry.length > 0) {
            return sendErrorResponse(res, "Work Schedule with this External Name already exists for the specified country", "Work Schedule with this External Name already exists for the specified country");
        }
        
        // Check is model exist in picklist
        const checkModel = await performQuery(
            `SELECT * FROM ${tables.picklist_master} WHERE (picklist_id = 18 and id = ?) AND is_deleted = 2`,
            [model]
        );
        if (checkModel.length === 0) {
            return sendErrorResponse(res, "Invalid model value", "Invalid model value");
        }

        // Check is time_recording_variant exist in picklist
        const checkTimeRecordingVarient = await performQuery(
            `SELECT * FROM ${tables.picklist_master} WHERE (picklist_id = 19 and id = ?) AND is_deleted = 2`,
            [time_recording_variant]
        );
        if (checkTimeRecordingVarient.length === 0) {
            return sendErrorResponse(res, "Invalid time recording varient value", "Invalid time recording varient value");
        }

        // Get current system time
        const systemTime = await getSystemTime();
        const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

        // Update existing work schedule record is_active from 1 to 2 to make it inactive
        await performQuery(
            `UPDATE ${tables.work_schedule} SET is_active = 2, updated_by = ?, updated_at = ? WHERE id = ?`,
            [req?.user?.id, currentTime, id]
        );

        // Insert new work schedule record with updated data
        const result = await performQuery(
            `INSERT INTO ${tables.work_schedule} SET ?`, {
                sequence_no: Number(existingRecord[0].sequence_no) + 1,
                schedule_id_external: existingRecord[0].schedule_id_external,
                external_name,
                starting_date,
                country,
                model,
                cross_midnight_allowed,
                user,
                average_hours_per_day: average_hours_per_day || null,
                average_hours_per_week: average_hours_per_week || null,
                average_hours_per_month: average_hours_per_month || null,
                average_hours_per_year: average_hours_per_year || null,
                average_working_days_per_week: average_working_days_per_week || null,
                shift_classification: shift_classification || null,
                time_recording_variant,
                created_by: req?.user?.id,
                created_at: currentTime,
                updated_by: req?.user?.id,
                updated_at: currentTime,
            }
        );

        // Send success response
        return sendResponse(res, {}, "Work Schedule Updated Successfully", 200);

    } catch (error) {
        console.log("Error While Updating Work Schedule: ", error);
        return sendErrorResponse(res, "Internal Server Error", "Internal Server Error");
    }
}



const DAY_NAMES = {
    0: "Sunday",
    1: "Monday",
    2: "Tuesday",
    3: "Wednesday",
    4: "Thursday",
    5: "Friday",
    6: "Saturday",
};

const TransformData = async (data) => {
  const map = new Map();

  data?.forEach((row) => {
    const key = `${row.id}`; // or `${row.id}__${row.schedule_id_external}` if needed

    if (!map.has(key)) {
      map.set(key, {
        id: row.id,
        schedule_id_external: row.schedule_id_external,
        external_name: row.external_name,
        starting_date: row.starting_date,
        country: row.country,
        model: row.model,
        model_value: row.model_value,
        cross_midnight_allowed: row.cross_midnight_allowed,
        user: row.user,
        average_hours_per_day: row.average_hours_per_day,
        average_hours_per_week: row.average_hours_per_week,
        average_hours_per_month: row.average_hours_per_month,
        average_hours_per_year: row.average_hours_per_year,
        average_working_days_per_week: row.average_working_days_per_week,
        shift_classification: row.shift_classification,
        time_recording_variant: row.time_recording_variant,
        time_recording_variant_value: row.time_recording_variant_value,
        created_at: row.created_at,
        updated_at: row.updated_at,
        created_by: row.created_by,
        updated_by: row.updated_by,
        work_schedule_details: [],
      });
    }

    const agg = map.get(key);

    // ----- Work schedule detail (day) -----
    if (row.work_schedule_detail_id != null) {
      // find the correct day object (create if missing)
      let detail = agg.work_schedule_details.find(
        (d) => d.id === row.work_schedule_detail_id
      );

      if (!detail) {
        detail = {
          id: row.work_schedule_detail_id,
          day: row.day,
          day_name: DAY_NAMES[row.day] ?? "",
          schedule_id_external: row.schedule_id_external,
          day_model_id_external: row.day_model_id_external,
          day_model_category: row.day_model_category,
          day_model_category_value: row.day_model_category_value,
          segments: [],
        };
        agg.work_schedule_details.push(detail);
      }

      // ----- Segment -----
      // Prefer day_model_segment_id (your desired output uses that),
      // otherwise fallback to segment_id
      const segId = row.day_model_segment_id ?? row.segment_id;

      const hasSegment =
        segId != null &&
        row.start_time != null &&
        row.end_time != null;

      if (hasSegment) {
        const exists = detail.segments.some((s) => s.id === segId);

        if (!exists) {
          detail.segments.push({
            id: segId,
            day_model_name: row.day_model_name,
            start_time: row.start_time,
            end_time: row.end_time,
            duration: row.duration,
            day_model_segments_category: row.day_model_segments_category,
            day_model_segment_category_value: row.day_model_segment_category_value,
          });
        }
      }
    }
  });

  // optional: sort days
  const result = Array.from(map.values());
  result.forEach((s) =>
    s.work_schedule_details.sort((a, b) => (a.day ?? 0) - (b.day ?? 0))
  );

  return result;
};
