const { sendErrorResponse, sendResponse, } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");

// --> Create Time Type Request
module.exports.createTimeTypeRequest = async (req, res) => {
  uploadedFiles = [];
  try {
    const {
      employee_id_external,
      time_type_id_external,
      available_balance,
      start_date,
      end_date,
      comment
    } = req.body;

    console.log("req.body: ", req.body);
    

    if (!employee_id_external || !time_type_id_external || !start_date || !end_date || !available_balance) {
      return sendErrorResponse(res, "Missing required fields!", "Missing required fields!");
    }

    
    // File upload
    let baseURL = `${req.protocol}://${req.get("host")}/uploads`;
    const attachment_path = req.files?.attachment_path? req.files.attachment_path[0].filename: null;
    
    if (attachment_path) uploadedFiles.push(attachment_path);
    const attachment_URL = attachment_path? `${baseURL}/${attachment_path}`: null;
    
    // Duplicate active request check (same employee + time type)
    const existing = await performQuery(`
      SELECT * FROM ${tables.time_type_request}
      WHERE employee_id_external = ?
      AND time_type_id_external = ? 
      AND (
        (STR_TO_DATE(?,'%Y-%m-%d') BETWEEN STR_TO_DATE(start_date, '%Y-%m-%d') AND STR_TO_DATE(end_date, '%Y-%m-%d'))
        OR
        (STR_TO_DATE(?,'%Y-%m-%d') BETWEEN STR_TO_DATE(start_date, '%Y-%m-%d') AND STR_TO_DATE(end_date, '%Y-%m-%d'))   
      )
       AND is_active = 1`,
      [employee_id_external, time_type_id_external, start_date, end_date, start_date, end_date]
    );

    if (existing.length > 0) {
      return sendErrorResponse(
        res,
        "Active Time Type Request already exists",
        "Active Time Type Request already exists"
      );
    }


    // Get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert Time Type Request
    await performQuery(
      `INSERT INTO ${tables.time_type_request} SET ?`,
      {
        sequence_no: 1,
        employee_id_external,
        time_type_id_external,
        available_balance,
        start_date,
        end_date,
        comment,
        attachment_path: attachment_URL,
        created_by: req?.user?.id,
        created_at: currentTime,
        updated_by: req?.user?.id,
        updated_at: currentTime
      }
    );

    return sendResponse(res, {}, "Time Type Request created successfully", 200);

  } catch (error) {
    console.error("Error Creating Time Type Request", error);
    return sendErrorResponse(res, error, "Error while creating Time Type Request");
  }
};

// --> Get Time Type Request
module.exports.getTimeTypeRequest = async (req, res) => {
  try {
    const { id, employee_id_external } = req.query;

    let query = `
      SELECT
        ttr.id,
        ttr.sequence_no,
        ttr.employee_id_external,
        p.display_name AS employee_name,
        ttr.time_type_id_external,
        tt.external_name AS time_type_name,
        ttr.available_balance,
        ttr.start_date,
        ttr.end_date,
        ttr.comment,
        ttr.attachment_path,
        ttr.approval_status,
        ttr.approval_remarks,
        ttr.approver_id_external
      FROM ${tables.time_type_request} ttr
      LEFT Join ${tables.time_type} tt ON ttr.time_type_id_external = tt.time_type_id_external and tt.is_active = 1
      LEFT Join ${tables.emp_employee} emp ON ttr.employee_id_external = emp.employee_id_external and emp.is_deleted = 2
      LEFT Join ${tables.per_person} p ON emp.person_id_external = p.person_id_external and p.is_deleted = 2
      WHERE ttr.is_active = 1
    `;

    if (id) {
      query += ` AND ttr.id = ${id}`;
    }

    if (employee_id_external) {
      query += ` AND ttr.employee_id_external = '${employee_id_external}'`;
    }

    query += ` ORDER BY ttr.id DESC`;

    const result = await performQuery(query);

    return sendResponse(
      res,
      result,
      "Time Type Request fetched successfully",
      200
    );

  } catch (error) {
    console.error("Error Fetching Time Type Request", error);
    return sendErrorResponse(res, error, "Error while fetching Time Type Request");
  }
};


// --> General Update (Employee updates pending request)
module.exports.updateTimeTypeRequestGeneral = async (req, res) => {
  try {
    const { id, comment, attachment } = req.body;
    if (!id) return sendErrorResponse(res, "Request ID required", "Request ID required");

    const existing = await performQuery(
      `SELECT * FROM ${tables.time_type_request} WHERE id = ? AND is_active = 1`,
      [id]
    );
    if (existing.length === 0) return sendErrorResponse(res, "Request not found", "Request not found");

    // Only pending requests can be updated
    if (existing[0].approval_status != 1)
      return sendErrorResponse(res, "Only pending request can be updated", "Only pending request can be updated");

    // Get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Deactivate old record
    await performQuery(
      `UPDATE ${tables.time_type_request} SET ? WHERE id = ?`,
      [{ is_active: 2, updated_by: req?.user?.id, updated_at: currentTime }, id]
    );

    // Handle file upload
    let attachment_URL = existing[0].attachment_path;
    if (req.files?.resignation_doc) {
      const fileName = req.files.resignation_doc[0].filename;
      attachment_URL = `${req.protocol}://${req.get("host")}/uploads/${fileName}`;
    }

    // Insert new version
    await performQuery(
      `INSERT INTO ${tables.time_type_request} SET ?`,
      {
        sequence_no: existing[0].sequence_no + 1,
        employee_id_external: existing[0].employee_id_external,
        time_type_id_external: existing[0].time_type_id_external,
        available_balance: existing[0].available_balance,
        start_date: existing[0].start_date,
        end_date: existing[0].end_date,
        approval_remarks: existing[0].approval_remarks,
        approver_id_external: existing[0].approver_id_external,
        comment,
        attachment_path: attachment_URL,
        approval_status: existing[0].approval_status, 
        is_active: 1,
        created_by: req?.user?.id,
        created_at: currentTime,
        updated_by: req?.user?.id,
        updated_at: currentTime
      }
    );

    return sendResponse(res, {}, "Request updated successfully", 200);

  } catch (error) {
    console.error(error);
    return sendErrorResponse(res, error, "Error updating request");
  }
};

// --> Approver Update (Approve / Reject)
module.exports.updateTimeTypeRequestApprover = async (req, res) => {
  try {
    const { id, status, approval_remarks } = req.body; // status 2=Approved, 3=Rejected
    if (!id || !status) return sendErrorResponse(res, "ID and status required", "ID and status required");
    if (![2,3].includes(Number(status))) return sendErrorResponse(res, "Invalid status", "Invalid status");

    const existing = await performQuery(
      `SELECT * FROM ${tables.time_type_request} WHERE id = ? AND is_active = 1`,
      [id]
    );
    if (existing.length === 0) return sendErrorResponse(res, "Request not found", "Request not found");

    if (existing[0].approval_status != 1)
      return sendErrorResponse(res, "Only pending requests can be updated by approver", "Only pending requests can be updated by approver");

    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Deactivate old record
    await performQuery(
      `UPDATE ${tables.time_type_request} SET ? WHERE id = ?`,
      [{ is_active: 2, updated_by: req?.user?.id, updated_at: currentTime }, id]
    );

    // Insert new version with updated status
    await performQuery(
      `INSERT INTO ${tables.time_type_request} SET ?`,
      {
        sequence_no: existing[0].sequence_no + 1,
        employee_id_external: existing[0].employee_id_external,
        time_type_id_external: existing[0].time_type_id_external,
        available_balance: existing[0].available_balance,
        start_date: existing[0].start_date,
        end_date: existing[0].end_date,
        approval_remarks,
        approver_id_external: existing[0].approver_id_external,
        comment: existing[0].comment,
        attachment_path: existing[0].attachment_path,
        approval_status: Number(status),
        is_active: 1,
        created_by: req?.user?.id,
        created_at: currentTime,
        updated_by: req?.user?.id,
        updated_at: currentTime
      }
    );

    return sendResponse(res, {}, "Request status updated successfully", 200);

  } catch (error) {
    console.error(error);
    return sendErrorResponse(res, error, "Error updating request status");
  }
};
