const { sendErrorResponse, sendResponse, } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");
const { last } = require("lodash");

// --> Create Time Profile
module.exports.createTimeProfile = async (req, res) => {
      try {
    const {
      time_profile_id_external,
      extrenal_name,
      effective_start_date,
      main_absence_time_type,
      main_attendance_time_type,
      main_break_time_type,
      country,
      time_recording_variant
    } = req.body;

    if (
      !time_profile_id_external ||
      !extrenal_name ||
      !effective_start_date ||
      !country ||
      !time_recording_variant
    ) {
      return sendErrorResponse(res, "Missing required fields!", "Missing required fields!");
    }

    // Duplicate check
    const existing = await performQuery(
      `SELECT * FROM ${tables.time_profile}
       WHERE time_profile_id_external = ? AND is_active = 1`,
      [time_profile_id_external]
    );
    if (existing.length > 0) {
      return sendErrorResponse(res, "Time Profile already exists", "Time Profile already exists");
    }

    // Picklist check
    const checkVariant = await performQuery(
      `SELECT * FROM ${tables.picklist_master}
       WHERE picklist_id = 19 AND id = ? AND is_deleted = 2`,
      [time_recording_variant]
    );
    if (checkVariant.length === 0) {
      return sendErrorResponse(res, "Invalid time recording variant", "Invalid time recording variant");
    }

    // Get current system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    await performQuery(
      `INSERT INTO ${tables.time_profile} SET ?`,
      {
        sequence_no: 1,
        time_profile_id_external,
        extrenal_name,
        effective_start_date,
        main_absence_time_type,
        main_attendance_time_type,
        main_break_time_type,
        country,
        time_recording_variant,
        created_by: req?.user?.id,
        created_at: currentTime,
        updated_by: req?.user?.id,
        updated_at: currentTime
      }
    );

    return sendResponse(res, {}, "Time Profile created successfully", 200);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while creating Time Profile");
  }
};

// --> Get Time Profile
module.exports.getTimeProfile = async (req, res) => {
    try {
    const { id, time_profile_id_external } = req.query;

    let query = `
      SELECT
        tp.*,
        mwt.external_name AS main_attendance_time_type_name,
        mat.external_name AS main_absence_time_type_name,
        mbt.external_name AS main_break_time_type_name,
        tp.created_at AS tp_created_at,
        tp.created_by AS tp_created_by,
        tp.updated_at AS tp_updated_at,
        tp.updated_by AS tp_updated_by,
        pm.picklist_option AS time_recording_variant_value,
        tpd.id AS time_profile_detail_id,
        tpd.time_type_id_external, tpd.external_code,
        tpdtt.external_name AS time_type_name,
        tpdtt.country AS time_type_country, tpdtt.unit AS time_type_unit, tpdtt.classification AS time_type_classification, 
        ttu.picklist_option AS time_type_unit_value,
        ttc.picklist_option AS time_type_classification_value,
        tpdtt.permitted_fractions_unit_day AS time_type_permitted_fractions_unit_day,
        tpdtt.permitted_fractions_unit_hour AS time_type_permitted_fractions_unit_hour,
        tpd.sequence_no,
        tpd.created_at, tpd.created_by,
        tpd.updated_at, tpd.updated_by
      FROM ${tables.time_profile} tp
      LEFT JOIN ${tables.picklist_master} pm ON tp.time_recording_variant = pm.id AND pm.picklist_id = 19 AND pm.is_deleted = 2
      LEFT JOIN ${tables.time_profile_details} tpd ON tp.time_profile_id_external = tpd.time_profile_id_external AND tpd.is_active = 1
      LEFT JOIN ${tables.time_type} tpdtt ON tpd.time_type_id_external = tpdtt.time_type_id_external AND tpdtt.is_active = 1
      LEFT JOIN ${tables.picklist_master} ttu ON tpdtt.unit = ttu.id AND ttu.picklist_id = 25 AND ttu.is_deleted = 2
      LEFT JOIN ${tables.picklist_master} ttc ON tpdtt.classification = ttc.id AND ttc.picklist_id = 22 AND ttc.is_deleted = 2
      LEFT JOIN ${tables.time_type} mwt ON tp.main_attendance_time_type = mwt.time_type_id_external AND mwt.is_active = 1
      LEFT JOIN ${tables.time_type} mat ON tp.main_absence_time_type = mat.time_type_id_external AND mat.is_active = 1
      LEFT JOIN ${tables.time_type} mbt ON tp.main_break_time_type = mbt.time_type_id_external AND mbt.is_active = 1
      WHERE tp.is_active = 1
    `;

    if (id) query += ` AND tp.id = ${id}`;
    if (time_profile_id_external)
      query += ` AND tp.time_profile_id_external = '${time_profile_id_external}'`;

    query += ` ORDER BY tp.id DESC`;

    const result = await performQuery(query);
    const transformedResult = await TransformData(result);
    return sendResponse(res, transformedResult, "Time Profile fetched successfully", 200);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while fetching Time Profile");
  }
};

// --> Update Time Profile
module.exports.updateTimeProfile = async (req, res) => {
     try {
    const {
      id,
      extrenal_name,
      effective_start_date,
      main_absence_time_type,
      main_attendance_time_type,
      main_break_time_type,
      country,
      time_recording_variant
    } = req.body;

    if (!id || !extrenal_name || !effective_start_date || !country || !time_recording_variant) {
      return sendErrorResponse(res, "Missing required fields!", "Missing required fields!");
    }

    const existing = await performQuery(
      `SELECT * FROM ${tables.time_profile} WHERE id = ?`,
      [id]
    );
    if (existing.length === 0) {
      return sendErrorResponse(res, "Time Profile not found", "Time Profile not found");
    }

    // Picklist check
    const checkVariant = await performQuery(
      `SELECT * FROM ${tables.picklist_master}
       WHERE picklist_id = 19 AND id = ? AND is_deleted = 2`,
      [time_recording_variant]
    );
    if (checkVariant.length === 0) {
      return sendErrorResponse(res, "Invalid time recording variant", "Invalid time recording variant");
    }

    // Get current system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    await performQuery(
      `UPDATE ${tables.time_profile} SET ? WHERE id = ?`,
      [{ is_active: 2, updated_by: req?.user?.id, updated_at: currentTime }, id]
    );

    await performQuery(
      `INSERT INTO ${tables.time_profile} SET ?`,
      {
        sequence_no: Number(existing[0].sequence_no) + 1,
        time_profile_id_external: existing[0].time_profile_id_external,
        extrenal_name,
        effective_start_date,
        main_absence_time_type,
        main_attendance_time_type,
        main_break_time_type,
        country,
        time_recording_variant,
        created_by: req?.user?.id,
        created_at: currentTime,
        updated_by: req?.user?.id,
        updated_at: currentTime
      }
    );

    return sendResponse(res, {}, "Time Profile updated successfully", 200);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while updating Time Profile");
  }
};




const TransformData = async (data) => {
  const map = new Map();
  data?.forEach((row) => {
    // Grouping key: adjust if you want a different grouping logic
    const key = `${row.id}`;
    if (!map.has(key)) {
      map.set(key, {
        id: row.id,
        time_profile_id_external: row.time_profile_id_external,
        extrenal_name: row.extrenal_name,
        effective_start_date: row.effective_start_date,
        main_attendance_time_type: row.main_attendance_time_type,
        main_attendance_time_type_name: row.main_attendance_time_type_name,
        main_absence_time_type: row.main_absence_time_type,
        main_absence_time_type_name: row.main_absence_time_type_name,
        main_break_time_type: row.main_break_time_type,
        main_break_time_type_name: row.main_break_time_type_name,
        country: row.country,
        time_recording_variant_value: row.time_recording_variant_value,
        created_at: row.tp_created_at,
        created_by: row.tp_created_by,
        updated_at: row.tp_updated_at,
        updated_by: row.tp_updated_by,
        time_profile_detail: [],
      });
    }
    const agg = map.get(key);
    if (row.time_profile_detail_id !== null) {
      const existingTimeProfileDetailId = agg.time_profile_detail.some(item => item.id === row.time_profile_detail_id);
      if (!existingTimeProfileDetailId) {
        agg.time_profile_detail.push({
          id: row.time_profile_detail_id,
          external_code: row.external_code,
          time_profile_id_external: row.time_profile_id_external,
          time_type_id_external: row.time_type_id_external,     
          external_name: row.time_type_name,     
          unit: row.time_type_unit,
          unit_value: row.time_type_unit_value,
          country: row.time_type_country,
          classification: row.time_type_classification,
          classification_value: row.time_type_classification_value,
          permitted_fractions_unit_day: row.time_type_permitted_fractions_unit_day,
          permitted_fractions_unit_hour: row.time_type_permitted_fractions_unit_hour,
        });
      }
    }
  });
  return Array.from(map.values());
};