const { sendErrorResponse, sendResponse, } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");
const { get } = require("lodash");

// --> Create Time Events
module.exports.createTimeEvents = async (req, res) => {
    try {
        const{
            date,
            employee_id_external,
            time_value,
            time_event_type,
            reason,
            comments,
        } = req.body;

        if (!date ||
            !employee_id_external ||
            !time_value ||
            !time_event_type || !reason
        ) {
            return sendErrorResponse(res, "Missing required fields", "Missing required fields");
        }

        // check for employee_id_external existence
        const employeeRecord = await performQuery(
            `SELECT * FROM ${tables.emp_employee} WHERE employee_id_external = ? AND is_deleted = 2`,
            [employee_id_external]
        );
        if (employeeRecord.length === 0) {
            return sendErrorResponse(res, "Employee not found", "Employee not found");
        }
        
        // check for time_event_type from picklist
        const timeEventTypeRecord = await performQuery(
            `SELECT * FROM ${tables.picklist_master} WHERE id = ? AND picklist_id = 26 AND is_deleted = 2`,
            [time_event_type]
        );
        if (timeEventTypeRecord.length === 0) {
            return sendErrorResponse(res, "Invalid time event type", "Invalid time event type");
        }

        // check for reason from picklist
        const reasonRecord = await performQuery(
            `SELECT * FROM ${tables.picklist_master} WHERE id = ? AND picklist_id = 27 AND is_deleted = 2`,
            [reason]
        );
        if (reasonRecord.length === 0) {
            return sendErrorResponse(res, "Invalid reason", "Invalid reason");
        }

        console.log("timeEventTypeRecord[0].picklist_option: ", timeEventTypeRecord[0].picklist_option);
        

        const time_event_id_external = `${employee_id_external}-${date}_${time_event_type == 103 ? 'IN' : 'OUT'}`;

        const existingRecord = await performQuery(
            `SELECT * FROM ${tables.time_event} WHERE time_event_id_external = ?`,
            [time_event_id_external]
        );
        if (existingRecord.length > 0) {
            return sendErrorResponse(res, "Time event already exists", "Time event already exists");
        }

        const systemTime = await getSystemTime();
        const currentTimestamp = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

        const insetQuery = `INSERT INTO ${tables.time_event} SET ?`;
        const values = {
            time_event_id_external,
            date,
            employee_id_external,
            time_value,
            time_event_type,
            reason,
            comments,
            created_by: req?.user?.id,
            created_at: currentTimestamp,
            updated_by: req?.user?.id,
            updated_at: currentTimestamp
        };
        const result = await performQuery(insetQuery, values);
        
        return sendResponse(res, 200, "Time event created successfully", 201);

    } catch (error) {
        console.error("Error creating time event:", error);
        return sendErrorResponse(res, "Internal server error", "Internal server error");
    }
};

// --> Get Time Events
module.exports.getTimeEvents = async (req, res) => {
    try {
        const {id, employee_id_external} = req.query;
        let query = `
            SELECT 
                te.id,
                te.date,
                te.employee_id_external,
                p.display_name AS employee_name,
                te.time_value,
                te.time_event_type,
                tet.picklist_option AS time_event_type_name,
                te.reason,
                tr.picklist_option AS reason_name,
                te.comments,
                te.approval_status,
                te.approver,
                app_by_p.display_name AS approver_name,
                te.approval_timestamp,
                te.approver_comments
            FROM ${tables.time_event} te
            LEFT JOIN ${tables.emp_employee} emp ON te.employee_id_external = emp.employee_id_external AND emp.is_deleted = 2
            LEFT Join ${tables.per_person} p ON emp.person_id_external = p.person_id_external and p.is_deleted = 2
            LEFT JOIN ${tables.emp_employee} app_by ON te.approver = app_by.employee_id_external AND app_by.is_deleted = 2
            LEFT Join ${tables.per_person} app_by_p ON app_by.person_id_external = app_by_p.person_id_external and app_by_p.is_deleted = 2
            LEFT JOIN ${tables.picklist_master} tet ON te.time_event_type = tet.id AND tet.picklist_id = 26 AND tet.is_deleted = 2
            LEFT JOIN ${tables.picklist_master} tr ON te.reason = tr.id AND tr.picklist_id = 27 AND tr.is_deleted = 2 
            WHERE te.is_active = 1
        `;

        if (id) {query += ` AND id = ${id}`;}
        if (employee_id_external) {query += ` AND te.employee_id_external = '${employee_id_external}'`;}
        query += ` ORDER BY te.id DESC`;

        const records = await performQuery(query);
        return sendResponse(res, records, "Time events fetched successfully", 200);

    } catch (error) {
        console.error("Error fetching time events:", error);
        return sendErrorResponse(res, "Internal server error", "Internal server error");
    }
};

// --> Update Time Events
module.exports.updateTimeEvents= async (req, res) => {
    try {
        const { 
            id, 
            date,
            time_value,
            time_event_type,
            reason,
            comments,
        } = req.body;
        
        if (
            !id ||
            !date ||
            !time_value ||
            !time_event_type || !reason
        ) {
            return sendErrorResponse(res, "Missing required field: id", "Missing required field: id");
        }

        const existingRecord = await performQuery(
            `SELECT * FROM ${tables.time_event} WHERE id = ? AND is_active = 1`,
            [id]
        );
        if (existingRecord.length === 0) {
            return sendErrorResponse(res, "Time event not found", "Time event not found");
        }

        // check for time_event_type from picklist
        const timeEventTypeRecord = await performQuery(
            `SELECT * FROM ${tables.picklist_master} WHERE id = ? AND picklist_id = 26 AND is_deleted = 2`,
            [time_event_type]
        );
        if (timeEventTypeRecord.length === 0) {
            return sendErrorResponse(res, "Invalid time event type", "Invalid time event type");
        }

        // check for reason from picklist
        const reasonRecord = await performQuery(
            `SELECT * FROM ${tables.picklist_master} WHERE id = ? AND picklist_id = 27 AND is_deleted = 2`,
            [reason]
        );
        if (reasonRecord.length === 0) {
            return sendErrorResponse(res, "Invalid reason", "Invalid reason");
        }
        
        // Get current system time        
        const systemTime = await getSystemTime();
        const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");        
        
        // Make existing record inactive
        const archiveData = await performQuery(
            `UPDATE ${tables.time_event} SET ? WHERE id = ?`,
            [{
                is_active: 2,
                updated_by: req?.user?.id,
                updated_at: currentTime
            },id]
        );
    
        // Insert new record with updated details
        const result = await performQuery(
            `INSERT INTO ${tables.time_event} SET  ?`,
            {
                time_event_id_external: existingRecord[0].time_event_id_external,
                employee_id_external: existingRecord[0].employee_id_external,
                sequence_no: Number(existingRecord[0].sequence_no) + 1,
                date: date || existingRecord[0].date,
                time_value: time_value || existingRecord[0].time_value,
                time_event_type: time_event_type || existingRecord[0].time_event_type,
                reason: reason || existingRecord[0].reason,
                comments: comments || existingRecord[0].comments,
                created_by: req?.user?.id,
                created_at: currentTime,
                updated_by: req?.user?.id,
                updated_at: currentTime
            }
        );

        return sendResponse(res, {}, "Time Events updated successfully", 200);
    
    } catch (error) {
        console.log("Error While Updating Time Events", error);
        return sendErrorResponse(res, error, "Error while updating time Events");
    }        
};

// --> Delete Time Events
module.exports.deleteTimeEvents = async (req, res) => {
    try {
        const { id } = req.body;

        if (!id) {
            return sendErrorResponse(res, "Missing required field: id", "Missing required field: id");
        }

        // Check if record exists and is active
        const existingRecord = await performQuery(
            `SELECT id FROM ${tables.time_event} WHERE id = ? AND is_active = 1`,
            [id]
        );

        if (existingRecord.length === 0) {
            return sendErrorResponse(res, "Time event not found", "Time event not found");
        }

        const systemTime = await getSystemTime();
        const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

        // Soft delete
        await performQuery(
            `UPDATE ${tables.time_event}
             SET is_active = 2, updated_by = ?, updated_at = ? WHERE id = ?`,
            [req?.user?.id || null, currentTime, id]
        );

        return sendResponse(res, {}, "Time event deleted successfully", 200);

    } catch (error) {
        console.error("Error While Deleting Time Events:", error);
        return sendErrorResponse(res,"Internal server error","Error while deleting time event"
        );
    }
};


// --> Update API for Approval / Rejection of Time Events


// --> Update Time Events
module.exports.timeEventApproval= async (req, res) => {
    try {
        const { 
            id, 
            approval_status,
            approver_comments,
        } = req.body;
        
        if (
            !id || !approval_status
        ) {
            return sendErrorResponse(res, "Missing required field: id, approval_status", "Missing required field: id, approval_status");
        }

        const existingRecord = await performQuery(
            `SELECT * FROM ${tables.time_event} WHERE id = ? AND is_active = 1`,
            [id]
        );
        if (existingRecord.length === 0) {
            return sendErrorResponse(res, "Time event not found", "Time event not found");
        }
        
        // Get current system time        
        const systemTime = await getSystemTime();
        const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");        
        
        // Make existing record inactive
        const archiveData = await performQuery(
            `UPDATE ${tables.time_event} SET ? WHERE id = ?`,
            [{
                approval_status,
                approver: req?.user?.employee_id_external,
                approval_timestamp: currentTime,
                approver_comments,
                updated_by: req?.user?.id,
                updated_at: currentTime
            },id]
        );

        return sendResponse(res, {}, "Time Events updated successfully", 200);
    
    } catch (error) {
        console.log("Error While Updating Time Events", error);
        return sendErrorResponse(res, error, "Error while updating time Events");
    }        
};