const { sendErrorResponse, sendResponse } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");

module.exports.createDayModelSegments = async (req, res) => {
  try {
    const {
      day_model_id_external,
      start_time,
      end_time,
      duration,
      day_model_segments_category,
    } = req.body;

    if (
      !day_model_id_external ||
      !start_time ||
      !end_time ||
      !duration ||
      !day_model_segments_category
    ) {
      const errorMessage = "Missing required fields!";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Duplicate check (same day model)
    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.day_model_segments}
       WHERE day_model_id_external = ? and day_model_segments_category = ?
       AND is_active = 1`,
      [day_model_id_external, day_model_segments_category]
    );
    if (existingRecord.length > 0) {
      return sendErrorResponse(
        res,
        "Day Model Segment already exists for this Category",
        "Day Model Segment already exists for this Category"
      );
    }

    // Picklist check
    const checkCategory = await performQuery(
      `SELECT * FROM ${tables.picklist_master}
       WHERE picklist_id = 20 AND id = ? AND is_deleted = 2`,
      [day_model_segments_category]
    );
    if (checkCategory.length === 0) {
      return sendErrorResponse(
        res,
        "Invalid day model segment category",
        "Invalid day model segment category"
      );
    }

    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    await performQuery(`INSERT INTO ${tables.day_model_segments} SET ?`, {
      sequence_no: 1,
      day_model_id_external,
      start_time,
      end_time,
      duration,
      day_model_segments_category,
      created_by: req?.user?.id,
      created_at: currentTime,
      updated_by: req?.user?.id,
      updated_at: currentTime,
    });

    return sendResponse(res, {}, "Day Model Segment created successfully", 200);
  } catch (error) {
    console.error("Error While Creating Day Model Segment", error);
    return sendErrorResponse(
      res,
      error,
      "Error while creating Day Model Segment"
    );
  }
};

// --> Get Day Model Segments
module.exports.getDayModelSegments = async (req, res) => {
  try {
    const { id, day_model_id_external } = req.query;

    let query = `
      SELECT
        dms.id,
        dms.sequence_no,
        dms.day_model_id_external,
        dm.external_name,
        dms.start_time,
        dms.end_time,
        dms.duration,
        dms.day_model_segments_category,
        pm.picklist_option AS day_model_segments_category_value
      FROM ${tables.day_model_segments} dms
      LEFT JOIN ${tables.picklist_master} pm
        ON dms.day_model_segments_category = pm.id
        AND pm.picklist_id = 20
        AND pm.is_deleted = 2
      LEFT JOIN ${tables.day_model} dm
        ON dms.day_model_id_external = dm.day_model_id_external
      WHERE dms.is_active = 1
    `;

    if (id) query += ` AND dms.id = ${id}`;
    if (day_model_id_external)
      query += ` AND dms.day_model_id_external = '${day_model_id_external}'`;

    query += ` ORDER BY dms.id DESC`;

    const result = await performQuery(query);
    return sendResponse(
      res,
      result,
      "Day Model Segments fetched successfully",
      200
    );
  } catch (error) {
    console.error("Error While Fetching Day Model Segments", error);
    return sendErrorResponse(
      res,
      error,
      "Error While Fetching Day Model Segments"
    );
  }
};

// --> Update Day Model Segments
module.exports.updateDayModelSegments = async (req, res) => {
  try {
    const { id, start_time, end_time, duration, day_model_segments_category } =
      req.body;

    if (
      !id ||
      !start_time ||
      !end_time ||
      !duration ||
      !day_model_segments_category
    ) {
      const errorMessage = "Missing required fields!";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.day_model_segments} WHERE id = ? AND is_active = 1`,
      [id]
    );
    if (existingRecord.length === 0) {
      return sendErrorResponse(
        res,
        "Day Model Segment not found",
        "Day Model Segment not found"
      );
    }

    const day_model_id_external = existingRecord[0].day_model_id_external;

    // Duplicate start_time check
    const duplicateRecord = await performQuery(
      `SELECT * FROM ${tables.day_model_segments}
       WHERE day_model_id_external = ?
       AND day_model_segments_category = ?
       AND is_active = 1
       AND id <> ?`,
      [day_model_id_external, day_model_segments_category, id]
    );
    if (duplicateRecord.length > 0) {
      return sendErrorResponse(
        res,
        "Day Model Segment already exists for this Category",
        "Day Model Segment already exists for this Category"
      );
    }

    // Picklist check
    const checkCategory = await performQuery(
      `SELECT * FROM ${tables.picklist_master}
       WHERE picklist_id = 20 AND id = ? AND is_deleted = 2`,
      [day_model_segments_category]
    );
    if (checkCategory.length === 0) {
      return sendErrorResponse(
        res,
        "Invalid day model segment category",
        "Invalid day model segment category"
      );
    }

    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Deactivate old record
    await performQuery(
      `UPDATE ${tables.day_model_segments} SET ? WHERE id = ?`,
      [
        {
          is_active: 2,
          updated_by: req?.user?.id,
          updated_at: currentTime,
        },
        id,
      ]
    );

    // Insert new record
    await performQuery(`INSERT INTO ${tables.day_model_segments} SET ?`, {
      sequence_no: Number(existingRecord[0].sequence_no) + 1,
      day_model_id_external,
      start_time,
      end_time,
      duration,
      day_model_segments_category,
      created_by: req?.user?.id,
      created_at: currentTime,
      updated_by: req?.user?.id,
      updated_at: currentTime,
    });

    return sendResponse(res, {}, "Day Model Segment updated successfully", 200);
  } catch (error) {
    console.error("Error While Updating Day Model Segment", error);
    return sendErrorResponse(
      res,
      error,
      "Error while updating Day Model Segment"
    );
  }
};
