const { sendErrorResponse, sendResponse, } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");
const fs = require('fs');
const path = require('path');
const upload = require("../../middleware/upload.js");
const e = require("express");

// API to Add Payment Details --> POST /personRecord/payment-details
module.exports.addPaymentDetails = async (req, res) => {
    const uploadedFiles = [];
    try {
        // Get variables from Request Body
        const {
            person_id,
            effective_start_date,
            pay_type,
            customer_type_external_code,
            payment_method_external_code,
            bank_country_code,
            account_owner,
            account_number,
            routing_number,
            bank_external_code,
            iban,
            business_identifier_code,
            currency_code,
            amount,
            percent,
            pay_sequence,
            purpose,
            external_code,
        } = req.body;

        // Validate required fields
        if (
            !person_id || !effective_start_date || !pay_type || !customer_type_external_code ||
            !payment_method_external_code || !account_owner || !account_number || !currency_code ||
            !amount || !percent || !pay_sequence || !purpose || !external_code || !bank_country_code ||
            !routing_number || !bank_external_code || !iban || !business_identifier_code
        ) {
            return sendErrorResponse(res, "All Fields are Required", "All Fields are Required", 400);
        }

        // Handle file uploads
        let baseURL = `${req.protocol}://${req.get("host")}/uploads`;
        const attachment_1 = req.files?.attachment_1 ? req.files.attachment_1[0].filename : null;
        if (attachment_1) uploadedFiles.push(attachment_1);
        if (!attachment_1) {
            return sendErrorResponse(res, "attachment is required", "attachment is required", 400);
        }

        // Check is person exists
        const personExists = await performQuery(`SELECT id FROM ${tables.per_person} WHERE id = ?`, [person_id]);
        if (personExists.length === 0) {
            return sendErrorResponse(res, "Person not found", "Person not Fount", 404);
        }

        // check is pay_type, customer_type_external_code or payment_method_external_code valid
        const TypeExists = await performQuery(
            `SELECT * FROM ${tables.picklist_master} WHERE (id = ? OR id = ? OR id = ?) AND is_deleted = 2`,
            [pay_type, customer_type_external_code, payment_method_external_code]
        );
        if (TypeExists.length === 0) {
            return sendErrorResponse(res, "Invalid payment type", "Invalid Payment Type", 400);
        }

        // check for existing external_code
        const existingPayment = await performQuery(
            `SELECT * from ${tables.per_payment_info_bank_details} Where external_code = ? AND is_deleted = 2`,
            [external_code]
        );
        if (existingPayment.length > 0) {
            return sendErrorResponse(res, "Payment Details with this external code already exists", "Duplicate Entry", 400);
        }

        // Calculate completion %
        const total_fields = 19; 
        let filled_fields = 0;
        const checkFilled = (v) => { if (v) filled_fields++; };
        [
            person_id,
            effective_start_date,
            pay_type,
            customer_type_external_code,
            payment_method_external_code,
            bank_country_code,
            account_owner,
            account_number,
            routing_number,
            bank_external_code,
            iban,
            business_identifier_code,
            currency_code,
            amount,
            percent,
            pay_sequence,
            purpose,
            external_code,
            attachment_1
        ].forEach(checkFilled);

        const completion = Math.round((filled_fields / total_fields) * 100);
        const is_completed = completion === 100 ? 2 : 1; // 2 = completed, 1 = not completed


        // Get system time
        const systemTime = await getSystemTime();
        const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

        // Create File URL
        const attachment1URL = attachment_1 ? `${baseURL}/${attachment_1}` : null;

        // Insert Educational Details Record
        const insertEducationResult = await performQuery(
            `INSERT INTO ${tables.per_payment_info_bank_details} SET ?`,
            {
                person_id: person_id,
                effective_start_date: effective_start_date,
                pay_type: pay_type,
                customer_type_external_code: customer_type_external_code,
                payment_method_external_code: payment_method_external_code,
                bank_country_code: bank_country_code,
                account_owner: account_owner,
                account_number: account_number,
                routing_number: routing_number,
                bank_external_code: bank_external_code,
                iban: iban,
                business_identifier_code: business_identifier_code,
                currency_code: currency_code,
                amount: amount,
                percent: percent,
                pay_sequence: pay_sequence,
                purpose: purpose,
                external_code: external_code,
                attachment_1: attachment1URL,
                completion: completion,
                is_completed: is_completed,
                created_at: currentTime,
                created_by: req?.user?.id,
                updated_at: currentTime,
                updated_by: req?.user?.id,
            }
        );

        // Send Success Response
        return sendResponse(res, [], "Payment Details added successfully", 201);

    } catch (error) {
        console.error("Error adding payment details:", error);
        return sendErrorResponse(res, "Failed to add payment details", "Failed to add payment details", 500);
    }
};


// API to Get Payment Details --> GET /personRecord/payment-details
module.exports.getPaymentDetails = async (req, res) => {
    try {
        // Get person_id from query parameters
        const { person_id, person_id_external } = req.query;

        // Create Select Query
        let selectQuery = `
            SELECT 
                pay.*, per.display_name, per.person_id_external,
                pt.picklist_option AS pay_type_value,
                cc.picklist_option AS customer_type_value,
                pm.picklist_option AS payment_method_value 
            FROM ${tables.per_payment_info_bank_details} pay
            Left Join ${tables.per_person} per ON pay.person_id = per.id
            Left Join ${tables.picklist_master} pt ON pt.id = pay.pay_type
            Left Join ${tables.picklist_master} cc ON cc.id = pay.customer_type_external_code
            Left Join ${tables.picklist_master} pm ON pm.id = pay.payment_method_external_code	
            WHERE pay.is_deleted = 2`;

        // Apply filters
        if (person_id) {
            selectQuery += ` AND pay.person_id = ${person_id}`;
        }
        if (person_id_external) {
            selectQuery += ` AND per.person_id_external = '${person_id_external}'`;
        }

        console.log(selectQuery);
        

        // Execute Query
        const paymentDetails = await performQuery(selectQuery);

        // Send Success Response
        return sendResponse(res, paymentDetails, "Payment Details fetched successfully", 200);

    } catch (error) {
        console.error("Error fetching payment details:", error);
        return sendErrorResponse(res, "Failed to fetch payment details", "Failed to fetch payment details", 500);
    }
};

// API to Update Payment Details --> PUT /personRecord/payment-details
module.exports.updatePaymentDetails = async (req, res) => {
    uploadedFiles = [];
    try {
        // Get variables from Request Body
        const {
            id,
            effective_start_date,
            pay_type,
            customer_type_external_code,
            payment_method_external_code,
            bank_country_code,
            account_owner,
            account_number,
            routing_number,
            bank_external_code,
            iban,
            business_identifier_code,
            currency_code,
            amount,
            percent,
            pay_sequence,
            purpose,
            external_code,
        } = req.body;

        // Validate required fields
        if (
            !id || !effective_start_date || !pay_type || !customer_type_external_code ||
            !payment_method_external_code || !account_owner || !account_number || !currency_code ||
            !amount || !percent || !pay_sequence || !purpose || !external_code || !bank_country_code ||
            !routing_number || !bank_external_code || !iban || !business_identifier_code
        ) {
            return sendErrorResponse(res, "All Fields are Required", "All Fields are Required", 400);
        }

        // Handle file uploads
        let baseURL = `${req.protocol}://${req.get("host")}/uploads`;
        const attachment_1 = req.files?.attachment_1 ? req.files.attachment_1[0].filename : null;
        if (attachment_1) uploadedFiles.push(attachment_1);
    
        // check for existing external_code
        const existingPayment = await performQuery(
            `SELECT * from ${tables.per_payment_info_bank_details} Where external_code = ? AND id <> ? AND is_deleted = 2`,
            [external_code, id]
        );
        if (existingPayment.length > 0) {
            return sendErrorResponse(res, "Payment Details with this external code already exists", "Duplicate Entry", 400);
        }

        // Get Existing Payment Details Record
        const existingRecords = await performQuery(
            `SELECT * FROM ${tables.per_payment_info_bank_details} WHERE id = ? AND is_deleted = 2`,
            [id]
        );
        if (existingRecords.length === 0) {
            return sendErrorResponse(res, "Payment Details record not found", "Payment Details record not found", 404);
        }
        const existingRecord = existingRecords[0];

        // Calculate completion %
        const total_fields = 19; 
        let filled_fields = 0;
        const checkFilled = (v) => { if (v) filled_fields++; };
        [
            existingRecord.person_id,
            effective_start_date || existingRecord.effective_start_date,
            pay_type || existingRecord.pay_type,
            customer_type_external_code || existingRecord.customer_type_external_code,
            payment_method_external_code || existingRecord.payment_method_external_code,
            bank_country_code || existingRecord.bank_country_code,
            account_owner || existingRecord.account_owner,
            account_number || existingRecord.account_number,
            routing_number || existingRecord.routing_number,
            bank_external_code || existingRecord.bank_external_code,
            iban || existingRecord.iban,
            business_identifier_code || existingRecord.business_identifier_code,
            currency_code || existingRecord.currency_code,
            amount || existingRecord.amount,
            percent || existingRecord.percent,
            pay_sequence || existingRecord.pay_sequence,
            purpose || existingRecord.purpose,
            external_code || existingRecord.external_code,
            attachment_1 || existingRecord.attachment_1
        ].forEach(checkFilled);

        const completion = Math.round((filled_fields / total_fields) * 100);
        const is_completed = completion === 100 ? 2 : 1; // 2 = completed, 1 = not completed

        // Get system time
        const systemTime = await getSystemTime();
        const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

        // Create File URL
        const attachment1URL = attachment_1 ? `${baseURL}/${attachment_1}` : existingRecord.attachment_1;

        // Mark existing record as deleted to maintain history
        const updateResult = await performQuery(
            `UPDATE ${tables.per_payment_info_bank_details} SET ? WHERE id = ?`,
            [
                {
                    is_deleted: 1,
                    updated_at: currentTime,
                    updated_by: req?.user?.id,
                },
                id
            ]
        );

        // Insert New Payment Details Record
        const insertPaymentResult = await performQuery(
            `INSERT INTO ${tables.per_payment_info_bank_details} SET ?`,
            {
                person_id: existingRecord.person_id,
                effective_start_date: effective_start_date || existingRecord.effective_start_date,
                pay_type: pay_type || existingRecord.pay_type,
                customer_type_external_code: customer_type_external_code || existingRecord.customer_type_external_code,
                payment_method_external_code: payment_method_external_code || existingRecord.payment_method_external_code,
                bank_country_code: bank_country_code || existingRecord.bank_country_code,
                account_owner: account_owner || existingRecord.account_owner,
                account_number: account_number || existingRecord.account_number,
                routing_number: routing_number || existingRecord.routing_number,
                bank_external_code: bank_external_code || existingRecord.bank_external_code,
                iban: iban || existingRecord.iban,
                business_identifier_code: business_identifier_code || existingRecord.business_identifier_code,
                currency_code: currency_code || existingRecord.currency_code,
                amount: amount || existingRecord.amount,
                percent: percent || existingRecord.percent,
                pay_sequence: pay_sequence || existingRecord.pay_sequence,
                purpose: purpose || existingRecord.purpose,
                external_code: external_code || existingRecord.external_code,
                attachment_1: attachment1URL || existingRecord.attachment_1,
                completion: completion,
                is_completed: is_completed,
                created_at: currentTime,
                created_by: req?.user?.id,
                updated_at: currentTime,
                updated_by: req?.user?.id,
            }
        );

        // Send Success Response
        return sendResponse(res, [], "Payment Details updated successfully", 200);

    } catch (error) {
        console.error("Error updating payment details:", error);
        return sendErrorResponse(res, "Failed to update payment details", "Failed to update payment details", 500);
    }
}

// API to Delete Payment Details --> DELETE /personRecord/payment-details
module.exports.deletePaymentDetails = async (req, res) => {
    try {
        // Get id from query parameters
        const { id } = req.query;

        // Validate id
        if (!id) {
            return sendErrorResponse(res, "Payment Details ID is required", "Payment Details ID is required", 400);
        }

        // Get Existing Payment Details Record
        const existingRecords = await performQuery(
            `SELECT * FROM ${tables.per_payment_info_bank_details} WHERE id = ? AND is_deleted = 2`,  
            [id]
        );
        if (existingRecords.length === 0) {
            return sendErrorResponse(res, "Payment Details record not found", "Payment Details record not found", 404);
        }

        // Get system time
        const systemTime = await getSystemTime();
        const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

        // Mark record as deleted
        const updateResult = await performQuery(
            `UPDATE ${tables.per_payment_info_bank_details} SET ? WHERE id = ?`,
            [
                {
                    is_deleted: 1,
                    updated_at: currentTime,
                    updated_by: req?.user?.id,
                },
                id
            ]
        );  

        // Send Success Response
        return sendResponse(res, [], "Payment Details deleted successfully", 200);

    } catch (error) {
        console.error("Error deleting payment details:", error);
        return sendErrorResponse(res, "Failed to delete payment details", "Failed to delete payment details", 500);
    }
};