const {
  sendErrorResponse,
  sendResponse,
} = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getTimezone, getSystemTime } = require("../../functions/getTimezone.js");
const fs = require('fs');
const path = require('path');


// CREATE Driving License
module.exports.createDrivingLicense = async (req, res) => {
  const uploadedFiles = [];
  try {
    const {
      person_id,
      country,
      license_no,
      issue_date,
      issue_date_hijri,
      expiry_date,
      expiry_date_hijri
    } = req.body;

    if (!person_id || !country || !license_no || !issue_date || !expiry_date) {
      const errorMessage = "Person ID, Country, License No, Issue Date, and Expiry Date are required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    
    // Handle file uploads
    let baseURL = `${req.protocol}://${req.get("host")}/uploads`;
    const frontPic = req.files?.attachment_1 ? req.files.attachment_1[0].filename : null;
    const backPic = req.files?.attachment_2 ? req.files.attachment_2[0].filename : null;
    if (frontPic) uploadedFiles.push(frontPic);
    if (backPic) uploadedFiles.push(backPic);
    
    // return error if either file is missing
    if (!frontPic || !backPic) {
      return sendErrorResponse(res, "Both front and back attachments are required", "Missing required Files", 400);
    }

    // check if person exists
    const personRecord = await performQuery(
      `SELECT * FROM ${tables.per_person} WHERE id = ? AND is_deleted = 2`,
      [person_id]
    );
    if (personRecord.length === 0) {
      const errorMessage = "Person not found";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // check if record already exists for person_id or license_no
    const existingRecords = await performQuery(
      `SELECT * FROM ${tables.per_driving_license} WHERE (person_id = ? OR license_no = ?) AND is_deleted = 2`,
      [person_id, license_no]
    );
    if (existingRecords.length > 0) {
      const errorMessage = "Driving license record already exists for this Person ID or License No";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Calculate completion and is_completed
    let total_fields = 9; // Total fields to consider for completion
    let filled_fields = 0;

    if (person_id) filled_fields++;
    if (country) filled_fields++;
    if (license_no) filled_fields++;
    if (issue_date) filled_fields++;
    if (issue_date_hijri) filled_fields++;
    if (expiry_date) filled_fields++;
    if (expiry_date_hijri) filled_fields++;
    if (frontPic) filled_fields++;
    if (backPic) filled_fields++;

    const completion = Math.round((filled_fields / total_fields) * 100);
    const is_completed = completion === 100 ? 2 : 1; // 2 for completed, 1 for not completed
    
    // Get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // create File URLs
    const frontPicURL = frontPic ? `${baseURL}/${frontPic}` : null;
    const backPicURL = backPic ? `${baseURL}/${backPic}` : null;

    // Prepare Insert Query
    const insertQuery = `
      INSERT INTO ${tables.per_driving_license} SET ?
    `;

    const insertParams = {
      person_id: person_id, 
      country: country, 
      license_no: license_no, 
      issue_date: issue_date, 
      issue_date_hijri: issue_date_hijri,
      expiry_date: expiry_date,
      expiry_date_hijri: expiry_date_hijri,
      attachment_1: frontPicURL,
      attachment_2: backPicURL,
      completion: completion,
      is_completed: is_completed,
      created_at: currentTime,
      created_by: req?.user?.id,
      updated_at: currentTime,
      updated_by: req?.user?.id
    };

    // Execute Insert Query
    await performQuery(insertQuery, insertParams);

    // Send success response
    return sendResponse(res, {}, "Driving license created successfully", 201);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while creating driving license");
  }
};

// READ Driving License (by person_id or all)
module.exports.getDrivingLicense = async (req, res) => {
  try {
    const { person_id, person_id_external } = req.query;

    let query = `
      SELECT 
        pdl.*, 
        pp.display_name AS person_name, pp.person_id_external,
        c.display_name AS created_by_name,
        u.display_name AS updated_by_name
        FROM ${tables.per_driving_license} AS pdl 
        LEFT JOIN ${tables.per_person} AS pp ON pdl.person_id = pp.id
        LEFT JOIN ${tables.per_person} AS c ON pdl.created_by = c.id
        LEFT JOIN ${tables.per_person} AS u ON pdl.updated_by = u.id
        WHERE pdl.is_deleted = 2`;

    if (person_id) {
      query += ` AND pdl.person_id = ${person_id}`;
    }
    else if (person_id_external) {
      query += ` AND pp.person_id_external = '${person_id_external}'`;
    }

    // Sort by most recent by id
    query += ` ORDER BY pp.id DESC`;

    const licenses = await performQuery(query);
    return sendResponse(res, licenses, "Driving license(s) fetched successfully", 200);
  } catch (error) {
    return sendErrorResponse(res, error, "Error while fetching driving license");
  }
};

// UPDATE Driving License
module.exports.updateDrivingLicense = async (req, res) => {
  const uploadedFiles = [];
  try {
    const {
      id,
      country,
      license_no,
      issue_date,
      issue_date_hijri,
      expiry_date,
      expiry_date_hijri,
    } = req.body;

    if (!id || !country || !license_no || !issue_date || !expiry_date) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    
    // Handle file uploads
    let baseURL = `${req.protocol}://${req.get("host")}/uploads`;
    const frontPic = req.files?.attachment_1 ? req.files.attachment_1[0].filename : null;
    const backPic = req.files?.attachment_2 ? req.files.attachment_2[0].filename : null;
    if (frontPic) uploadedFiles.push(frontPic);
    if (backPic) uploadedFiles.push(backPic);

    // Check if license_no already exists for other records
    const existingRecords = await performQuery(
      `SELECT * FROM ${tables.per_driving_license} WHERE license_no = ? AND id <> ? AND is_deleted = 2`,
      [license_no, id]
    );
    if (existingRecords.length > 0) {
      const errorMessage = "Another driving license record already exists with this License No";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // get Previous Record
    const existingRecord = await performQuery(`
      SELECT * FROM ${tables.per_driving_license} WHERE id = ?`, [id]
    );
    if (!existingRecord || existingRecord.length === 0) {
      const errorMessage = "Driving license not found";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    const previousRecord = existingRecord[0];

    // Calculate completion and is_completed
    let total_fields = 9; // Total fields to consider for completion
    let filled_fields = 0;

    if (previousRecord.person_id) filled_fields++;
    if (country || previousRecord.country) filled_fields++;
    if (license_no || previousRecord.license_no) filled_fields++;
    if (issue_date || previousRecord.issue_date) filled_fields++;
    if (issue_date_hijri || previousRecord.issue_date_hijri) filled_fields++;
    if (expiry_date || previousRecord.expiry_date) filled_fields++;
    if (expiry_date_hijri || previousRecord.expiry_date_hijri) filled_fields++;
    if (frontPic || previousRecord.attachment_1) filled_fields++;
    if (backPic || previousRecord.attachment_2) filled_fields++;

    const completion = Math.round((filled_fields / total_fields) * 100);
    const is_completed = completion === 100 ? 2 : 1; // 2 for completed, 1 for not completed


    // Get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // create File URLs
    const frontPicURL = frontPic ? `${baseURL}/${frontPic}` : null;
    const backPicURL = backPic ? `${baseURL}/${backPic}` : null;

    // Mark existing Record as deleted to maintain history
    const updateExistingRecord = await performQuery(`
      UPDATE ${tables.per_driving_license} SET ? WHERE id = ?`, [
      {
        is_deleted: 1,
        updated_at: currentTime,
        updated_by: req?.user?.id,
      }, id]
    );

    // create New Record with updated data
    const newRecord = await performQuery(
      `INSERT INTO ${tables.per_driving_license} SET ?`, {
        person_id: previousRecord.person_id, 
        country: country || previousRecord.country, 
        license_no: license_no || previousRecord.license_no,
        issue_date: issue_date || previousRecord.issue_date, 
        issue_date_hijri: issue_date_hijri || previousRecord.issue_date_hijri,
        expiry_date: expiry_date || previousRecord.expiry_date,
        expiry_date_hijri: expiry_date_hijri || previousRecord.expiry_date_hijri,
        attachment_1: frontPicURL || previousRecord.attachment_1,
        attachment_2: backPicURL || previousRecord.attachment_2,
        completion: completion,
        is_completed: is_completed,
        created_at: currentTime,  
        created_by: req?.user?.id,
        updated_at: currentTime,
        updated_by: req?.user?.id
      }
    );

    // Send success response
    return sendResponse(res, {}, "Driving license updated successfully", 200);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while updating driving license");
  }
};

// DELETE Driving License (soft delete)
module.exports.deleteDrivingLicense = async (req, res) => {
  try {
    const { id } = req.query;

    if (!id) {
      const errorMessage = "License ID is required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    const deleteQuery = `
      UPDATE ${tables.per_driving_license}
      SET ? WHERE id = ?
    `;
    await performQuery(deleteQuery, [{is_deleted: 1, updated_at: currentTime, updated_by: req?.user?.id}, id]);

    return sendResponse(res, {}, "Driving license deleted successfully", 200);
  } catch (error) {
    return sendErrorResponse(res, error, "Error while deleting driving license");
  }
};
