const { generateOTP } = require("../services/OTP.js");
const { sendMail } = require("../services/MAIL.js");
const bcrypt = require("bcryptjs");
const jwt = require("jsonwebtoken");
const {
  sendErrorResponse,
  sendResponse,
  generatepassword_reset_token,
} = require("../utils/index.js");
const { tables } = require("../utils/tables.js");
const { JWT_SECRET } = require("../config/index.js");
const { performQuery } = require("../utils/db.js");
const moment = require("moment-timezone");
const { getTimezone, getSystemTime } = require("../functions/getTimezone.js");

module.exports.signUpApplicant = async (req, res) => {
  // Extracting applicant details from request body
  try {
    const { salutation, first_name, mid_name, last_name, email, password } = req.body;
    // Validate the email and password
    if (!salutation || !first_name || !last_name || !email || !password) {
      return sendErrorResponse(
        res,
        "Salutation, First Name, Last Name, Email and Password are required",
        "Salutation, First Name, Last Name, Email and Password are required"
      );
    }

    // Check if the email address is already in use
    const selectEmailQuery = `SELECT email FROM ${tables.per_email} WHERE email = ? and is_deleted = 2`;
    const selectParams = [email];

    // Perform the database query
    const existingEmailRecord = await performQuery(selectEmailQuery, selectParams);
    if (existingEmailRecord?.length) {
      const errorMessage = "Email address is already in use";
      return sendErrorResponse(res, errorMessage, errorMessage);
    } 

    const strongPattern = /^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[!@#$%^&*]).{8,}$/;

    if (!strongPattern.test(password)) {
      const errorMessage = "Password is not strong enough";
      return sendErrorResponse(res, errorMessage, "Password is not strong enough. It must be at least 8 characters long and include uppercase letters, lowercase letters, numbers, and special characters.");
    }

    // Generate a one-time password (verification_otp) for email verification
    const verification_otp = generateOTP();
    
    // Hashing the password before storing it
    const hashedPassword = await bcrypt.hash(password, 10);

    // Get the current system time in the correct timezone
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    const personRecord = await performQuery(
      `INSERT INTO ${tables.per_person} SET ?`,
      {
        salutation: salutation,
        first_name: first_name,
        mid_name: mid_name,
        last_name: last_name,
        display_name: `${!salutation ? '' : salutation} ${first_name} ${!mid_name ? '' : mid_name + ' '} ${last_name}`,
        created_at: currentTime,
        created_by: null,
        updated_at: currentTime,
        updated_by: null
      }
    );

    const emailRecord = await performQuery(
      `INSERT INTO ${tables.per_email} SET ?`,
      {
        person_id : personRecord.insertId,
        email: email,
        email_type: 1,
        is_primary: 1,
        created_at: currentTime,
        created_by: null,
        updated_at: currentTime,
        updated_by: null
      }
    );

    const loginRecord = await performQuery(
      `INSERT INTO ${tables.per_login_data} SET ?`,
      {
        person_id: personRecord.insertId,
        email_id: emailRecord.insertId,
        password: hashedPassword,
        verification_otp: verification_otp,
        verification_otp_at: currentTime,
        is_new: 2,
        created_at: currentTime,
        created_by: null,
        updated_at: currentTime,
        updated_by: null
      }
    );
    
    // If the employee is created successfully, send a response
      const emailBody = `
        <!DOCTYPE html>
        <html>
          <head>
              <title>Verification Code</title>
          </head>
          <body style="font-family: Arial, sans-serif; background-color: #f4f4f4; padding: 20px;">
              <div style="max-width: 600px; margin: 0 auto; background-color: #ffffff; padding: 30px; border-radius: 10px;">
                <h1 style="color: #333333;">Hello ${salutation} ${first_name} ${!mid_name ? '' : mid_name} ${last_name},</h1>
                <p style="color: #666666;">Your verification code is "<b>${verification_otp}</b>".</p>
                <p style="color: #666666;">Please enter this code to verify your account.</p>
                <p style="color: #666666; font-size: 12px;">If you did not request this code, please ignore this email.</p>
              </div>
          </body>
        </html>
      `;
      // Send the verification_otp to the employee's email address
      await sendMail(email, "Verify your email address", emailBody);

      return sendResponse(
        res,
        [],
        "Applicant registered successfully",
        201
      );
  } catch (error) {
    return sendErrorResponse(res, error, "Error while creating employee");
  }
};


module.exports.verifyEmail = async (req, res) => {
  // Extracting verification_otp from request body
  try {
    const { email, verification_otp } = req.body;

    if (!verification_otp || !email) {
      const errorMessage = "Email and verification_otp are required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Find the employee by email address
    const applicantData = await performQuery(
      `
        SELECT 
          pe.id, pe.person_id, 
          ld.verification_otp, ld.verification_otp_at, 
          pp.person_id_external, pp.first_name, pp.mid_name, pp.last_name, pp.display_name
        FROM ${tables.per_email} AS pe
        left join ${tables.per_login_data} ld on pe.person_id = ld.person_id
        left join ${tables.per_person} pp on pe.person_id = pp.id
        WHERE pe.email = ? and pe.is_deleted = 2`,
      [email]
    );

    if (!applicantData?.length) {
      const errorMessage = "Applicant does not exist";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check if the verification_otp is valid
    if (verification_otp !== applicantData[0]?.verification_otp) {
      const errorMessage = "Invalid verification_otp";
      return sendErrorResponse(res, errorMessage, errorMessage, 400);
    }

    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");
    const applicant = applicantData[0];

    // Verify the applicant's email address
    const updateEmailDB = await performQuery(
      `UPDATE ${tables.per_email} SET ? where id = ?`,
      [{
        is_verified: 1,
        updated_at: currentTime,
        updated_by: applicant?.person_id
      }, applicant?.id]
    );


    // Generate a JWT token for the employee
    const payload = {
      id: applicant?.person_id_external,
      name: applicant?.display_name,
      email: applicant?.email,
      role: applicant?.role,
    };

    const token = await jwt.sign(payload, JWT_SECRET, {});    
    return sendResponse(
      res,
      { token },
      "Email Verification Successfully and loged in",
      201
    );
  } catch (error) {
    return sendErrorResponse(res, error, "Email Verification Failed");
  }
};



module.exports.loginApplicant = async (req, res) => {
  try {
    // Extracting email and password from request body
    const { email, password } = req.body;

    // Validate email and password
    if (!email || !password) {
      const errorMessage = "Email and password are required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Added ld.is_new in query to check password setup status
    const selectApplicantQuery = `
      SELECT 
        pe.id AS email_id,
        pe.person_id,
        pe.email,
        pe.is_verified,
        ld.id AS login_id,
        ld.password,
        ld.is_two_fa,
        ld.is_new,                -- 🔧 Added
        pp.salutation,
        pp.first_name,
        pp.mid_name,
        pp.last_name,
        pp.display_name
      FROM ${tables.per_email} pe
      LEFT JOIN ${tables.per_login_data} ld ON pe.person_id = ld.person_id
      LEFT JOIN ${tables.per_person} pp ON pe.person_id = pp.id
      WHERE pe.email = ? AND pe.is_deleted = 2
    `;
    const applicant = await performQuery(selectApplicantQuery, [email]);

    // If applicant not found
    if (!applicant?.length) {
      return sendErrorResponse(res, {}, "Invalid email or password", 401);
    }
    const user = applicant[0];

    // Compare password
    const isPasswordValid = await bcrypt.compare(password, user.password);
    if (!isPasswordValid) {
      return sendErrorResponse(res, {}, "Invalid email or password", 401);
    }


    // Check if email is verified
    if (user.is_verified !== 1) {
      
      // Generate a one-time password (verification_otp) for email verification
      const verification_otp = generateOTP();

      // get System Time
      const systemTime = await getSystemTime();
      const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

      // Update Otp & its creation time in database
      const result = await performQuery(
        `UPDATE ${tables.per_login_data} SET verification_otp = ?, verification_otp_at = ?
         WHERE id = ?`,
        [verification_otp, currentTime, user.login_id]
      );
      
      // Send Verification Email
      const emailBody = `
        <!DOCTYPE html>
        <html>
          <head>
              <title>Verification Code</title>
          </head>
          <body style="font-family: Arial, sans-serif; background-color: #f4f4f4; padding: 20px;">
              <div style="max-width: 600px; margin: 0 auto; background-color: #ffffff; padding: 30px; border-radius: 10px;">
                <h1 style="color: #333333;">Hello ${user.salutation} ${user.display_name},</h1>
                <p style="color: #666666;">Your verification code is "<b>${verification_otp}</b>".</p>
                <p style="color: #666666;">Please enter this code to verify your account.</p>
                <p style="color: #666666; font-size: 12px;">If you did not request this code, please ignore this email.</p>
              </div>
          </body>
        </html>
      `;
      // Send the verification_otp to the employee's email address
      await sendMail(email, "Verify your email address", emailBody);

      return sendErrorResponse(res, {}, "Email is not verified", 400);
    }

    // Get current system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Added: Check if user needs to set a new password
    if (user.is_new === 1) {
      return sendResponse(
        res,
        { person_id: user.person_id, email: user.email },
        "Please set a new password before logging in.",
        200
      );
    }

    // Only proceed if is_new = 2
    if (user.is_new === 2) {
      // Check if 2FA is enabled
      if (user.is_two_fa == 1) {
        // Generate a one-time password (OTP)
        const otp = generateOTP();

        // Update OTP in database
        const updateQuery = `
          UPDATE ${tables.per_login_data}
          SET verification_otp = ?, verification_otp_at = ?
          WHERE person_id = ?
        `;
        await performQuery(updateQuery, [otp, currentTime, user.person_id]);

        // Prepare HTML email body
        const emailBody = `
        <!DOCTYPE html>
        <html>
          <head>
             <title>Login Verification</title>
          </head>
          <body style="font-family: Arial, sans-serif; background-color: #f4f4f4; padding: 20px;">
             <div style="max-width: 600px; margin: 0 auto; background-color: #ffffff; padding: 30px; border-radius: 10px;">
                <h1 style="color: #333333;">Hello ${user.display_name},</h1>
                <p style="color: #666666;">Your OTP is "<b>${otp}</b>".</p>
                <p style="color: #666666;">Please enter this code to verify your login.</p>
                <p style="color: #666666; font-size: 12px;">If you did not request this code, please ignore this email.</p>
              </div>
          </body>
        </html>
        `;

        // Send OTP email
        await sendMail(email, "Login Verification", emailBody);

        return sendResponse(
          res,
          {},
          "Please verify your login by entering the OTP sent to your email address",
          200
        );
      } else {
        // Generate JWT token
        const payload = {
          id: user.person_id,
          name: user.display_name,
          email: user.email,
          role: "applicant",
          iat: Math.floor(systemTime.valueOf() / 1000),
        };

        const token = await jwt.sign(payload, JWT_SECRET, {
          expiresIn: "24h",
        });

        // Send successful login response
        return sendResponse(
          res,
          { name: user.display_name, token: token },
          "Login successful",
          200
        );
      }
    }
  } catch (error) {
    console.log("Error: ", error);
    return sendErrorResponse(res, error, "Error while login");
  }
};

module.exports.loginVerifyApplicant = async (req, res) => {
  try {
    const { email, otp } = req.body;

    // Validate inputs
    if (!email || !otp) {
      const errorMessage = "Email and OTP are required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    const selectQuery = `
      SELECT 
        ld.person_id,
        ld.verification_otp,
        ld.verification_otp_at,
        pp.first_name,
        pp.mid_name,
        pp.last_name,
        pp.display_name,
        pe.email
      FROM ${tables.per_login_data} ld
      LEFT JOIN ${tables.per_person} pp ON ld.person_id = pp.id
      LEFT JOIN ${tables.per_email} pe ON ld.person_id = pe.person_id
      WHERE pe.email = ? AND ld.verification_otp = ? AND pe.is_deleted = 2
    `;
    const applicant = await performQuery(selectQuery, [email, otp]);

    if (!applicant?.length) {
      return sendErrorResponse(res, {}, "Invalid OTP", 400);
    }

    const user = applicant[0];

    // Check if OTP expired (valid 5 minutes)
    const otpTime = new Date(user.verification_otp_at);
    otpTime.setMinutes(otpTime.getMinutes() + 5);

    if (otpTime < new Date()) {
      return sendErrorResponse(res, {}, "OTP expired, please login again", 400);
    }

    // Get system time
    const systemTime = await getSystemTime();

    // Generate JWT token
    const payload = {
      id: user.person_id,
      name: user.display_name,
      email: user.email,
      role: "applicant",
      iat: Math.floor(systemTime.valueOf() / 1000),
    };

    const token = await jwt.sign(payload, JWT_SECRET, {
      expiresIn: "24h",
    });

    // Return success response
    return sendResponse(
      res,
      { name: user.display_name, token: token },
      "Email verified successfully",
      200
    );
  } catch (error) {
    console.log("Error while verifying login:", error);
    return sendErrorResponse(res, error, "Error while verifying login");
  }
};


module.exports.forgotPasswordApplicant = async (req, res) => {
  try {
    // Extract email from request body
    const { email } = req.body;

    if (!email) {
      const errorMessage = "Email is required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    const selectApplicantQuery = `
      SELECT 
        pe.person_id,
        pe.email,
        pp.first_name,
        pp.mid_name,
        pp.last_name,
        pp.display_name
      FROM ${tables.per_email} pe
      LEFT JOIN ${tables.per_person} pp ON pe.person_id = pp.id
      WHERE pe.email = ? AND pe.is_deleted = 2
    `;
    const applicant = await performQuery(selectApplicantQuery, [email]);

    // If not found
    if (!applicant?.length) {
      return sendErrorResponse(res, {}, "Applicant does not exist", 404);
    }

    const user = applicant[0];

    // Generate password reset token
    const resetToken = await generatepassword_reset_token();

    // Get current system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Update password reset token in login table
    const updateQuery = `
      UPDATE ${tables.per_login_data}
      SET password_reset_token = ?, password_reset_token_created_at = ?
      WHERE person_id = ?
    `;
    await performQuery(updateQuery, [resetToken, currentTime, user.person_id]);

    // Email body
    const emailBody = `
    <!DOCTYPE html>
    <html>
      <head>
         <title>Password Reset</title>
      </head>
      <body style="font-family: Arial, sans-serif; background-color: #f4f4f4; padding: 20px;">
         <div style="max-width: 600px; margin: 0 auto; background-color: #ffffff; padding: 30px; border-radius: 10px;">
            <h1 style="color: #333333;">Hello ${user.display_name},</h1>
            <p style="color: #666666;">Your password reset token is "<b>${resetToken}</b>".</p>
            <p style="color: #666666;">Use this token to reset your password.</p>
            <p style="color: #666666;">This token will expire in 30 minutes.</p>
            <p style="color: #666666; font-size: 12px;">If you did not request this, please ignore this email.</p>
          </div>
      </body>
    </html>
    `;

    // Send email
    await sendMail(email, "Reset your password", emailBody);

    // Success response
    return sendResponse(
      res,
      {},
      "Password reset token sent to your email address",
      200
    );
  } catch (error) {
    console.log("Error while forgot password:", error);
    return sendErrorResponse(res, error, "Error while forgot password");
  }
};

module.exports.resetPasswordApplicant = async (req, res) => {
  try {
    // Extract reset token and new password from request
    const { resetToken, newPassword } = req.body;

    if (!resetToken || !newPassword) {
      const errorMessage = "ResetToken and NewPassword are required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    const selectApplicantQuery = `
      SELECT 
        ld.person_id,
        ld.password_reset_token_created_at,
        pp.display_name
      FROM ${tables.per_login_data} ld
      LEFT JOIN ${tables.per_person} pp ON ld.person_id = pp.id
      WHERE ld.password_reset_token = ?
    `;
    const applicant = await performQuery(selectApplicantQuery, [resetToken]);

    // If not found
    if (!applicant?.length) {
      return sendErrorResponse(res, {}, "Invalid password reset token", 401);
    }

    const user = applicant[0];

    // Check if token expired (valid for 30 mins)
    const tokenCreatedAt = new Date(user.password_reset_token_created_at);
    tokenCreatedAt.setMinutes(tokenCreatedAt.getMinutes() + 30);

    if (tokenCreatedAt < new Date()) {
      return sendErrorResponse(
        res,
        {},
        "Token expired, please request a new password reset",
        401
      );
    }

    // Hash new password
    const hashedPassword = await bcrypt.hash(newPassword, 10);

    // Update applicant password and clear token
    const updateQuery = `
      UPDATE ${tables.per_login_data}
      SET password = ?, password_reset_token = NULL, password_reset_token_created_at = NULL
      WHERE person_id = ?
    `;
    await performQuery(updateQuery, [hashedPassword, user.person_id]);

    // Success response
    return sendResponse(res, {}, "Password reset successfully", 200);
  } catch (error) {
    console.log("Error while password reset:", error);
    return sendErrorResponse(res, error, "Error while password reset");
  }
};

module.exports.setNewPassword = async (req, res) => {
  try {
    // Get email and new password from request body
    const { email, newPassword } = req.body;

    // Validation 
    if (!email || !newPassword) {
      const errorMessage = "Email and New Password are required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    const strongPattern = /^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[!@#$%^&*]).{8,}$/;

    if (!strongPattern.test(newPassword)) {
      const errorMessage = "Password is not strong enough";
      return sendErrorResponse(res, errorMessage, "Password is not strong enough. It must be at least 8 characters long and include uppercase letters, lowercase letters, numbers, and special characters.");
    }

    const selectApplicantQuery = `
      SELECT 
        pe.person_id,
        pp.display_name
      FROM ${tables.per_email} pe
      LEFT JOIN ${tables.per_person} pp ON pe.person_id = pp.id
      WHERE pe.email = ? AND pe.is_deleted = 2
    `;
    const applicant = await performQuery(selectApplicantQuery, [email]);

    // If not found
    if (!applicant?.length) {
      return sendErrorResponse(res, {}, "Invalid email address", 401);
    }

    // Get person_id from applicant
    const { person_id } = applicant[0];

    // Hash new password
    const hashedPassword = await bcrypt.hash(newPassword, 10);

    // Update applicant password
    const updateQuery = `
      UPDATE ${tables.per_login_data}
      SET password = ?, is_new = 2
      WHERE person_id = ?
    `;
    await performQuery(updateQuery, [hashedPassword, person_id]);

    // Success response
    return sendResponse(res, {}, "New password set successfully", 200);


  } catch (error) {
    return sendErrorResponse(res, error, "Error while setting new password");
  }
}