const multer = require('multer');
const path = require('path');
const fs = require('fs');

// Ensure uploads folder exists

const UPLOAD_DIR = path.join(__dirname, '..', 'uploads');
const PICTURE_DIR = path.join(UPLOAD_DIR, 'picture');
const DOCUMENT_DIR = path.join(UPLOAD_DIR, 'document');

// Ensure directory structure exists at startup
for (const dir of [UPLOAD_DIR, PICTURE_DIR, DOCUMENT_DIR]) {
  if (!fs.existsSync(dir)) fs.mkdirSync(dir, { recursive: true });
}

// Decide subfolder by mimetype
const isImage = (m) => /^image\/(jpe?g|png|webp|gif)$/i.test(m);
const isPdf   = (m) => m === 'application/pdf';

// Storage configuration
const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, UPLOAD_DIR);
  },
  
  // destination: (req, file, cb) => {
  //   if (isImage(file.mimetype)) return cb(null, PICTURE_DIR);
  //   if (isPdf(file.mimetype))   return cb(null, DOCUMENT_DIR);
  //   cb(new Error('Unsupported file type'));
  // },
  filename: (req, file, cb) => {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1e9);
    const ext = path.extname(file.originalname);
    const base = path.basename(file.originalname, ext)
      .toLowerCase()
      .replace(/[^a-z0-9]+/gi, '-')
      .replace(/^-+|-+$/g, '');
    const fileName = `${base}-${uniqueSuffix}${ext}`;
    cb(null, fileName);
  }
});

// File filter
const fileFilter = (req, file, cb) => {
  const allowedTypes = ['application/pdf', 'image/jpeg', 'image/png', 'image/jpg'];
  if (allowedTypes.includes(file.mimetype)) {
    cb(null, true);
  } else {
    cb(new Error('Only PDF, JPEG, JPG, and PNG files are allowed'));
  }
};

// Multer configuration
const upload = multer({
  storage,
  fileFilter,
  limits: { fileSize: 5 * 1024 * 1024 } // 5MB
});

module.exports = upload;
