const {
  sendErrorResponse,
  sendResponse,
} = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getTimezone, getSystemTime } = require("../../functions/getTimezone.js");
const { isUndefined, create } = require("lodash");
const fs = require('fs');
const path = require('path');

// CREATE VISA IQAMA
module.exports.createVisaIqama = async (req, res) => {
  const uploadedFiles = []; 
  try {
    const {
      person_id,
      country,
      document_no,
      document_type,
      boarder_no,
      issue_date,
      issue_date_hijri,
      issue_place,
      issuing_authority,
      validated = 1, // default to 1 if not provided
      expiry_date,
      expiry_date_hijri,
      notes
    } = req.body;

    // required fields check
    if (
      isUndefined(person_id) || isUndefined(country) || isUndefined(document_no) || isUndefined(document_type) || 
      isUndefined(issue_date) || isUndefined(issue_place) || isUndefined(issuing_authority) ||
      isUndefined(expiry_date)
    ) {
      const errorMessage = "Person ID, Document No, Issue Date and Expiry Date are required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Handle file uploads
    let baseURL = `${req.protocol}://${req.get("host")}/uploads`;
    const attachment_1 = req.files?.attachment_1 ? req.files.attachment_1[0].filename : null;
    const attachment_2 = req.files?.attachment_2 ? req.files.attachment_2[0].filename : null;

    if (attachment_1) uploadedFiles.push(attachment_1);
    if (attachment_2) uploadedFiles.push(attachment_2);

    // Error if either file missing
    if (!attachment_1 && !attachment_2) {
      return sendErrorResponse(res, "Visa / Iqama Picture is required", "Visa / Iqama Picture is required", 400);
    }

    // check if person exists
    const personRecords = await performQuery(
      `SELECT * FROM ${tables.per_person} WHERE id = ? AND is_deleted = 2`,
      [person_id]
    );
    if (personRecords.length === 0) {
      const errorMessage = "Person not found";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // check if record already exists for person_id or document_no
    const existingRecords = await performQuery(
      `SELECT * FROM ${tables.per_visa_iqama} WHERE (person_id = ? OR document_no = ?) AND is_deleted = 2`,
      [person_id, document_no]
    );
    if (existingRecords.length > 0) {
      const errorMessage = "Visa Iqama record already exists for this person or document";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Calculate completion %
    let total_fields = 15;
    let filled_fields = 0;

    const checkFilled = (v) => { if (v) filled_fields++; };
    [
      person_id, 
      country, 
      document_no, 
      document_type, 
      boarder_no, 
      issue_date, 
      issue_date_hijri,
      issue_place, 
      issuing_authority, 
      validated,
      expiry_date, 
      expiry_date_hijri, 
      attachment_1, 
      attachment_2, 
      notes
    ].forEach(checkFilled);

    const completion = Math.round((filled_fields / total_fields) * 100);
    const is_completed = completion === 100 ? 2 : 1;  // 2 = completed, 1 = not completed

    // Get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");
                    
    // create File URLs
    const attachment_1_URL = attachment_1 ? `${baseURL}/${attachment_1}` : null;
    const attachment_2_URL = attachment_2 ? `${baseURL}/${attachment_2}` : null;

    // Insert Query
    const insertQuery = `INSERT INTO ${tables.per_visa_iqama} SET ?`;
    const insertParams = {
      person_id : person_id,
      country : country,
      document_no : document_no,
      document_type : document_type,
      boarder_no : boarder_no,
      issue_date : issue_date,
      issue_date_hijri : issue_date_hijri,
      issue_place : issue_place,
      issuing_authority : issuing_authority,
      validated: validated, 
      expiry_date : expiry_date,
      expiry_date_hijri : expiry_date_hijri,
      attachment_1: attachment_1_URL,
      attachment_2: attachment_2_URL,
      notes : notes,
      completion : completion,
      is_completed : is_completed,
      created_at: currentTime,
      created_by: req?.user?.id,
      updated_at: currentTime,
      updated_by: req?.user?.id
    };

    // Execute Insert Query
    await performQuery(insertQuery, insertParams);

    // Send success response
    return sendResponse(res, {}, "Visa Iqama created successfully", 201);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while creating Visa Iqama");
  }
};

// READ Visa Iqama (by person_id or all)
module.exports.getVisaIqama = async (req, res) => {
  try {
    const { person_id, person_id_external } = req.query;

    let query = `
      SELECT 
        pvi.*, 
        pp.display_name AS person_name, pp.person_id_external,
        c.display_name AS created_by_name,
        u.display_name AS updated_by_name
      FROM ${tables.per_visa_iqama} AS pvi 
      LEFT JOIN ${tables.per_person} AS pp ON pvi.person_id = pp.id
      LEFT JOIN ${tables.per_person} AS c ON pvi.created_by = c.id
      LEFT JOIN ${tables.per_person} AS u ON pvi.updated_by = u.id
      WHERE pvi.is_deleted = 2`;

    if (person_id) {
      query += ` AND pvi.person_id = ${person_id}`;
    } 

    else if (person_id_external) {
      query += ` AND pp.person_id_external = '${person_id_external}'`;
    }

    // sort by most recent id
    query += ` ORDER BY pvi.id DESC`;

    // Execute Query
    const visaIqama = await performQuery(query);

    // Response
    return sendResponse(res, visaIqama, "Visa Iqama record(s) fetched successfully", 200);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while fetching Visa Iqama");
  }
};

// UPDATE VISA IQAMA
module.exports.updateVisaIqama = async (req, res) => {
  const uploadedFiles = [];
  try {
    const {
      id,
      country,
      document_no,
      document_type,
      boarder_no,
      issue_date,
      issue_date_hijri,
      issue_place,
      issuing_authority,
      validated = 1, // default to 1 if not provided
      expiry_date,
      expiry_date_hijri,
      notes
    } = req.body;

    if (
      isUndefined(id) || isUndefined(country) || isUndefined(document_no) || isUndefined(document_type) || 
      isUndefined(issue_date) || isUndefined(issue_place) || isUndefined(issuing_authority) || isUndefined(expiry_date)
    ) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Handle file uploads
    let baseURL = `${req.protocol}://${req.get("host")}/uploads`;
    const attachment_1 = req.files?.attachment_1 ? req.files.attachment_1[0].filename : null;
    const attachment_2 = req.files?.attachment_2 ? req.files.attachment_2[0].filename : null;
    if (attachment_1) uploadedFiles.push(attachment_1);
    if (attachment_2) uploadedFiles.push(attachment_2);

    // Check if boarder no or document no already exists for other records
    const existingRecords = await performQuery(
      `SELECT * FROM ${tables.per_visa_iqama} WHERE (boarder_no = ? OR document_no = ?) AND id <> ? AND is_deleted = 2`,
      [boarder_no, document_no, id]
    );
    if (existingRecords.length > 0) {
      const errorMessage = "Another Visa Iqama record already exists with this Boarder No or Document No";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Get Previous Record
    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.per_visa_iqama} WHERE id = ? AND is_deleted = 2`,
      [id]
    );

    if (!existingRecord || existingRecord.length === 0) {
      const errorMessage = "Visa Iqama not found";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    const previousRecord = existingRecord[0];

    // Calculate completion and is_completed
    let total_fields = 15;
    let filled_fields = 0;

    const checkFilled = (v) => { if (v) filled_fields++; };
    [
      previousRecord.person_id,
      country || previousRecord.country,
      document_no || previousRecord.document_no,
      document_type || previousRecord.document_type,
      boarder_no || previousRecord.boarder_no,
      issue_date || previousRecord.issue_date,
      issue_date_hijri || previousRecord.issue_date_hijri,
      issue_place || previousRecord.issue_place,
      issuing_authority || previousRecord.issuing_authority,
      validated || previousRecord.validated,
      expiry_date || previousRecord.expiry_date,
      expiry_date_hijri || previousRecord.expiry_date_hijri,
      attachment_1 || previousRecord.attachment_1,
      attachment_2 || previousRecord.attachment_2,
      notes || previousRecord.notes
    ].forEach(checkFilled);

    const completion = Math.round((filled_fields / total_fields) * 100);
    const is_completed = completion === 100 ? 2 : 1;

    // Get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");
                    
    // create File URLs
    const attachment_1_URL = attachment_1 ? `${baseURL}/${attachment_1}` : null;
    const attachment_2_URL = attachment_2 ? `${baseURL}/${attachment_2}` : null;

    // Mark existing Record as deleted to maintain history
    await performQuery(
      `UPDATE ${tables.per_visa_iqama} SET ? WHERE id = ?`,
      [
        {
          is_deleted: 1,
          updated_at: currentTime,
          updated_by: req?.user?.id,
        },
        id,
      ]
    );

    // Insert new updated record
    await performQuery(
      `INSERT INTO ${tables.per_visa_iqama} SET ?`,
      {
        person_id: previousRecord.person_id,
        country: country || previousRecord.country,
        document_no: document_no || previousRecord.document_no,
        document_type: document_type || previousRecord.document_type,
        boarder_no: boarder_no || previousRecord.boarder_no,
        issue_date: issue_date || previousRecord.issue_date,
        issue_date_hijri: issue_date_hijri || previousRecord.issue_date_hijri,
        issue_place: issue_place || previousRecord.issue_place,
        issuing_authority: issuing_authority || previousRecord.issuing_authority,
        validated: validated || previousRecord.validated,
        expiry_date: expiry_date || previousRecord.expiry_date,
        expiry_date_hijri: expiry_date_hijri || previousRecord.expiry_date_hijri,
        attachment_1: attachment_1_URL || previousRecord.attachment_1,
        attachment_2: attachment_2_URL || previousRecord.attachment_2,
        notes: notes || previousRecord.notes,
        completion: completion,
        is_completed: is_completed,
        created_at: currentTime,
        created_by: req?.user?.id,
        updated_at: currentTime,
        updated_by: req?.user?.id,
      }
    );

    return sendResponse(res, {}, "Visa Iqama updated successfully", 200);

  } catch (error) {
    console.log("Error in Update Iqama/Visa: ", error);
    return sendErrorResponse(res, error, "Error while updating Visa Iqama");
  }
};

// DELETE VISA IQAMA (soft delete)
module.exports.deleteVisaIqama = async (req, res) => {
  try {
    const { id } = req.query;

    if (isUndefined(id)) {
      const errorMessage = "Visa Iqama ID is required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    const deleteQuery = `
      UPDATE ${tables.per_visa_iqama}
      SET ? WHERE id = ?
    `;

    await performQuery(deleteQuery, [
      { 
        is_deleted: 1,
        updated_at: currentTime,
        updated_by: req?.user?.id
      },
      id
    ]);

    return sendResponse(res, {}, "Visa Iqama deleted successfully", 200);
    
  } catch (error) {
    return sendErrorResponse(res, error, "Error while deleting Visa Iqama");
  }
};
