const {
  sendErrorResponse,
  sendResponse,
} = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getTimezone, getSystemTime } = require("../../functions/getTimezone.js");


// CREATE GOSI INFO
module.exports.createGosiInfo = async (req, res) => {
  try {
    const {
      person_id,
      social_insurance_grouping,
      blood_group,
      registered_in_gosi,
      registeration_date,
      end_date,
      gosi_no,
      non_register_reason,
      religion
    } = req.body;

    // required fields check
    if (
      !person_id || !social_insurance_grouping || !blood_group || !religion || !registered_in_gosi || 
      (registered_in_gosi == 1 ? (!registeration_date || !gosi_no || !end_date) : !non_register_reason)
    ) {
      return sendErrorResponse(res, "Missing required fields", "Missing required fields");
    }

    // Check if person exists
    const personCheck = await performQuery(
      `SELECT * FROM ${tables.per_person} WHERE id = ? AND is_deleted = 2`,
      [person_id]
    );
    if (personCheck.length === 0) {
      return sendErrorResponse(res, "Person not found", "Person not found");
    }

    // Check if record already exists
    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.per_specific_data_for_gosi} WHERE person_id = ? AND is_deleted = 2`,
      [person_id]
    );
    if (existingRecord.length > 0) {
      return sendErrorResponse(res, "GOSI record already exists for this Person ID", "Duplicate record");
    }

    // Calculate completion %
    const total_fields = 9;
    let filled_fields = 0;
    const checkFilled = (v) => { if (v) filled_fields++; };
    [
      person_id,
      social_insurance_grouping,
      blood_group,
      registered_in_gosi,
      registeration_date,
      end_date,
      gosi_no,
      non_register_reason,
      religion
    ].forEach(checkFilled);

    const completion = Math.round((filled_fields / total_fields) * 100);
    const is_completed = completion === 100 ? 2 : 1;

    // Get system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert query
    const insertQuery = `INSERT INTO ${tables.per_specific_data_for_gosi} SET ?`;
    const insertParams = {
      person_id : person_id,
      social_insurance_grouping : social_insurance_grouping,
      blood_group : blood_group,
      registered_in_gosi : registered_in_gosi,
      registeration_date : registeration_date,
      end_date : end_date,
      gosi_no : gosi_no,
      non_register_reason : non_register_reason,
      religion : religion,
      completion : completion,
      is_completed : is_completed,
      is_deleted: 2,
      created_at: currentTime,
      created_by: req?.user?.id,
      updated_at: currentTime,
      updated_by: req?.user?.id
    };

    // Execute Insert Query
    await performQuery(insertQuery, insertParams);

    // Send success response
    return sendResponse(res, {}, "GOSI info created successfully", 201);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while creating GOSI info");
  }
};

// READ Specific data for gosi (by person_id or all)
module.exports.getGosiInfo = async (req, res) => {
  try {
    const { person_id, person_id_external } = req.query;

    let query = `
      SELECT 
        pg.*,
        pp.display_name AS person_name, 
        pp.person_id_external
      FROM ${tables.per_specific_data_for_gosi} AS pg
      LEFT JOIN ${tables.per_person} AS pp ON pg.person_id = pp.id
      WHERE pg.is_deleted = 2
    `;

    if (person_id) {
      query += ` AND pg.person_id = ${person_id}`;
    } 
    else if (person_id_external) {
      query += ` AND pg.person_id_external = '${person_id_external}'`;
    }

    // Sort by most recent id
    query += ` ORDER BY pg.id DESC`;

    // Execute query
    const addressRecords = await performQuery(query);

    // Send response
    return sendResponse(res, addressRecords, "Address record(s) fetched successfully", 200);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while fetching Address records");
  }
};

// UPDATE GOSI INFO
module.exports.updateGosiInfo = async (req, res) => {
  try {
    const {
      id,
      social_insurance_grouping,
      blood_group,
      registered_in_gosi,
      registeration_date,
      end_date,
      gosi_no,
      non_register_reason,
      religion
    } = req.body;

    // Required field check
    if (
      !id || !social_insurance_grouping || !blood_group || !religion || !registered_in_gosi || 
      (registered_in_gosi == 1 ? (!registeration_date || !gosi_no || !end_date) : !non_register_reason)
    ) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // if gosi_no is provided check if it available in other record
    if (gosi_no) {
      // Check if GOSI number already exists (for another record)
      const existingGosiNo = await performQuery(
        `SELECT * FROM ${tables.per_specific_data_for_gosi} WHERE gosi_no = ? AND id <> ? AND is_deleted = 2`,
        [gosi_no, id]
      );
      if (existingGosiNo.length > 0) {
        const errorMessage = "Another record already exists with this GOSI No";
        return sendErrorResponse(res, errorMessage, errorMessage);
      }
    }

    // get Previous Record
    const existingRecord = await performQuery(`
      SELECT * FROM ${tables.per_specific_data_for_gosi} WHERE id = ?`, [id]
    );
    if (!existingRecord || existingRecord.length === 0) {
      const errorMessage = "GOSI record not found";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    const previousRecord = existingRecord[0];

    // Calculate completion %
    let total_fields = 9;
    let filled_fields = 0;

    const checkFilled = (v) => {
      if (v !== null && v !== undefined && v !== "") filled_fields++;
    };

    [
      previousRecord.person_id,
      social_insurance_grouping || previousRecord.social_insurance_grouping,
      blood_group || previousRecord.blood_group,
      registered_in_gosi || previousRecord.registered_in_gosi,
      registeration_date || previousRecord.registeration_date,
      registeration_date || previousRecord.end_date,
      gosi_no || previousRecord.gosi_no,
      non_register_reason || previousRecord.non_register_reason,
      religion || previousRecord.religion,
    ].forEach(checkFilled);

    const completion = Math.round((filled_fields / total_fields) * 100);
    const is_completed = completion === 100 ? 2 : 1;

    // System time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Mark old record as deleted (for history)
    await performQuery(
      `UPDATE ${tables.per_specific_data_for_gosi} SET ? WHERE id = ?`,
      [
        {
          is_deleted: 1,
          updated_at: currentTime,
          updated_by: req?.user?.id
        },
        id
      ]
    );

    // Insert new updated record
    await performQuery(
      `INSERT INTO ${tables.per_specific_data_for_gosi} SET ?`,
      {
        person_id: previousRecord.person_id,
        social_insurance_grouping: social_insurance_grouping || previousRecord.social_insurance_grouping,
        blood_group: blood_group || previousRecord.blood_group,
        registered_in_gosi: registered_in_gosi || previousRecord.registered_in_gosi,
        registeration_date: registeration_date || previousRecord.registeration_date,
        end_date: end_date || previousRecord.end_date,
        gosi_no: gosi_no || previousRecord.gosi_no,
        non_register_reason: non_register_reason || previousRecord.non_register_reason,
        religion: religion || previousRecord.religion,
        completion,
        is_completed,
        is_deleted: 2,
        created_at: currentTime,
        created_by: req?.user?.id,
        updated_at: currentTime,
        updated_by: req?.user?.id
      }
    );

    return sendResponse(res, {}, "GOSI record updated successfully", 200);
  } catch (error) {
    console.log("Error in updateGosi:", error);
    return sendErrorResponse(res, error, "Error while updating GOSI record");
  }
};

// DELETE GOSI INFO
module.exports.deleteGosiInfo = async (req, res) => {
  try {
    const { id } = req.body;

    if (!id) {
      const errorMessage = "Missing required field: id";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // System time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    await performQuery(
      `UPDATE ${tables.per_specific_data_for_gosi} SET ? WHERE id = ?`,
      [
        {
          is_deleted: 1,
          updated_at: currentTime,
          updated_by: req?.user?.id
        },
        id
      ]
    );

    // Send Response
    return sendResponse(res, {}, "GOSI record deleted successfully", 200);

  } catch (error) {
    console.log("Error in deleteGosi:", error);
    return sendErrorResponse(res, error, "Error while deleting GOSI record");
  }
};
