const {
  sendErrorResponse,
  sendResponse,
} = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");


// CREATE Section
module.exports.createSection = async (req, res) => {
  try {
    const {
      section_id_external,
      department_id,
      effective_start_date,
      description,
      description_arabic,
    } = req.body;

    if (!section_id_external || !department_id || !effective_start_date) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check department_id exists in department table
    const isDepartmentExist = await performQuery(
      `SELECT * FROM ${tables.org_department} WHERE id = ? AND is_deleted = 2`,
      [department_id]
    );

    if (isDepartmentExist.length === 0) {
      const errorMessage = "Department ID not found in Department table";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check for duplicate section_id_external
    const existingSection = await performQuery(
      `SELECT id FROM ${tables.org_section} WHERE section_id_external = ? AND is_deleted = 2`,
      [section_id_external]
    );
    if (existingSection && existingSection.length > 0) {
      const errorMessage = "Section ID already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Get system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert Record
    await performQuery(`INSERT INTO ${tables.org_section} SET ?`, {
      section_id_external,
      department_id,
      effective_start_date,
      description,
      description_arabic,
      created_at: currentTime,
      created_by: req?.user?.id,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    });

    return sendResponse(res, {}, "Section created successfully", 201);

  } catch (error) {
    console.error("Error while creating Section:", error);
    return sendErrorResponse(res, error, "Error while creating Section");
  }
};


// READ Section(s)
module.exports.getSection = async (req, res) => {
  try {
    const { id, section_id_external, department_id, department_id_external, division_id, division_id_external, business_unit_id, bu_id_external, entity_id, entity_id_external } = req.query;

    let query = `
      SELECT 
        s.*,
        department.department_id_external, department.description AS department_description,
        division.division_id_external, division.description AS division_description,
        bu.bu_id_external, bu.description AS business_unit_description,
        le.entity_id_external, le.name AS legal_entity_name
      FROM ${tables.org_section} s
      LEFT JOIN ${tables.org_department} department ON s.department_id = department.id AND department.is_deleted = 2
      LEFT JOIN ${tables.org_division} division ON department.division_id = division.id AND division.is_deleted = 2
      LEFT JOIN ${tables.org_business_unit} bu ON division.business_unit_id = bu.id AND bu.is_deleted = 2
      LEFT JOIN ${tables.org_legal_entity} le ON bu.entity_id = le.id AND le.is_deleted = 2
      WHERE s.is_deleted = 2
    `;

    if (id) query += ` AND s.id = ${id}`;
    if (section_id_external) query += ` AND s.section_id_external = '${section_id_external}'`;
    if (department_id) query += ` AND s.department_id = ${department_id}`;
    if (department_id_external) query += ` AND department.department_id_external = '${department_id_external}'`;
    if (division_id) query += ` AND division.id = ${division_id}`;
    if (division_id_external) query += ` AND division.division_id_external = '${division_id_external}'`;
    if (business_unit_id) query += ` AND bu.id = ${business_unit_id}`;
    if (bu_id_external) query += ` AND bu.bu_id_external = '${bu_id_external}'`;
    if (entity_id) query += ` AND le.id = ${entity_id}`;
    if (entity_id_external) query += ` AND le.entity_id_external = '${entity_id_external}'`;

    query += ` ORDER BY s.id DESC`;

    const sections = await performQuery(query);
    return sendResponse(res, sections, "Section(s) fetched successfully", 200);

  } catch (error) {
    console.error("Error while fetching Section(s):", error);
    return sendErrorResponse(res, error, "Error while fetching Section(s)");
  }
};


// UPDATE Section
module.exports.updateSection = async (req, res) => {
  try {
    const {
      id,
      section_id_external,
      department_id,
      effective_start_date,
      description,
      description_arabic,
      status,
    } = req.body;

    if (!id || !section_id_external || !department_id || !effective_start_date) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check if record exists
    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.org_section} WHERE id = ? AND is_deleted = 2`,
      [id]
    );
    if (!existingRecord || existingRecord.length === 0) {
      const errorMessage = "Section record not found";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    const oldRecord = existingRecord[0];

    // Check for duplicate section_id_external
    const duplicateCheck = await performQuery(
      `SELECT id FROM ${tables.org_section} WHERE section_id_external = ? AND id <> ? AND is_deleted = 2`,
      [section_id_external, id]
    );
    if (duplicateCheck && duplicateCheck.length > 0) {
      const errorMessage = "Section ID already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Update record
    await performQuery(`UPDATE ${tables.org_section} SET ? WHERE id = ?`, [{
      effective_start_date: effective_start_date || oldRecord.effective_start_date,
      description: description || oldRecord.description,
      description_arabic: description_arabic || oldRecord.description_arabic,
      status: status || oldRecord.status,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    }, id]);

    return sendResponse(res, {}, "Section updated successfully", 200);
  } catch (error) {
    console.error("Error while updating Section:", error);
    return sendErrorResponse(res, error, "Error while updating Section");
  }
};


// DELETE Section (Soft Delete)
// module.exports.deleteSection = async (req, res) => {
//   try {
//     const { id } = req.query;

//     if (!id) {
//       const errorMessage = "Section ID is required";
//       return sendErrorResponse(res, errorMessage, errorMessage);
//     }

//     // Get System Time
//     const systemTime = await getSystemTime();
//     const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

//     // Soft delete
//     await performQuery(
//       `UPDATE ${tables.org_section} SET ? WHERE id = ?`,
//       [
//         {
//           is_deleted: 1,
//           updated_at: currentTime,
//           updated_by: req?.user?.id,
//         },
//         id,
//       ]
//     );

//     return sendResponse(res, {}, "Section deleted successfully", 200);
//   } catch (error) {
//     console.error("Error while deleting Section:", error);
//     return sendErrorResponse(res, error, "Error while deleting Section");
//   }
// };
