const { sendErrorResponse, sendResponse } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");
const { error } = require("winston");

// CREATE Location
module.exports.createLocation = async (req, res) => {
  try {
    const {
      location_id_external,
      entity_id,
      location_group_id,
      location_name,
      effective_start_date,
      address_line_1,
      address_line_2,
      address_line_3,
      locality,
      dependent_locality,
      administrative_area,
      sub_administrative_area,
      postal_code,
      country_code,
      sorting_code,
      language_code,
    } = req.body;

    // Validation
    if (
      !location_id_external || !entity_id || !location_name || 
      !effective_start_date || !address_line_1 || !locality || 
      !administrative_area || !country_code
    ) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check for duplicate location_id_external
    const duplicateCheck = await performQuery(
      `SELECT id FROM ${tables.org_location} WHERE location_id_external = ? AND is_deleted = 2`,
      [location_id_external]
    );
    if (duplicateCheck && duplicateCheck.length > 0) {
      const errorMessage = "Location external ID already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check entity_id existence in legal entity table
    const isLegalEntity = await performQuery(
      `SELECT * FROM ${tables.org_legal_entity} WHERE id = ? AND is_deleted = 2`,
      [entity_id]
    );
    if (isLegalEntity.length === 0) {
      const errorMessage = "Legal Entity Not Found against entity_id";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    if(location_group_id){
      // Check location_group_id existence in location group table
      const isLocationGroup = await performQuery(
        `SELECT * FROM ${tables.org_location_group} WHERE id = ? AND is_deleted = 2`,
        [location_group_id]
      );
      if (isLocationGroup.length === 0) {
        const errorMessage = "Location Group Not Found against location_group_id";
        return sendErrorResponse(res, errorMessage, errorMessage);
      }
    }

    // System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert record
    await performQuery(`INSERT INTO ${tables.org_location} SET ?`, {
      location_id_external,
      entity_id: Number(entity_id),
      location_group_id: Number(location_group_id),
      location_name,
      effective_start_date,
      address_line_1,
      address_line_2,
      address_line_3,
      locality,
      dependent_locality,
      administrative_area,
      sub_administrative_area,
      postal_code,
      country_code,
      sorting_code,
      language_code,
      created_at: currentTime,
      created_by: req?.user?.id,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    });

    return sendResponse(res, {}, "Location created successfully", 201);
  } catch (error) {
    console.error("Error While Creating Location: ", error);
    return sendErrorResponse(res, error, "Error while creating location");
  }
};

// READ Location(s)
module.exports.getLocation = async (req, res) => {
  try {
    const { id } = req.query;

    let query = `
      SELECT *
      FROM ${tables.org_location}
      WHERE is_deleted = 2
    `;

    if (id) query += ` AND id = ${id}`;
    query += ` ORDER BY id DESC`;

    const result = await performQuery(query);
    return sendResponse(res, result, "Location(s) fetched successfully", 200);
  } catch (error) {
    console.error("Error While Fetching Location(s): ", error);
    return sendErrorResponse(res, error, "Error while fetching location(s)");
  }
};

// UPDATE Location
module.exports.updateLocation = async (req, res) => {
  try {
    const { 
      id,
      location_id_external,
      entity_id,
      location_group_id,
      location_name,
      effective_start_date,
      address_line_1,
      address_line_2,
      address_line_3,
      locality,
      dependent_locality,
      administrative_area,
      sub_administrative_area,
      postal_code,
      country_code,
      sorting_code,
      language_code, 
      status
    } = req.body;

    // Validation
    if (
      !location_id_external || !entity_id || !location_name || 
      !effective_start_date || !address_line_1 || !locality || 
      !administrative_area  || !country_code
    ) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.org_location} WHERE id = ? AND is_deleted = 2`,
      [id]
    );

    if (!existingRecord || existingRecord.length === 0) {
      const errorMessage = "Location not found";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    const oldRecord = existingRecord[0];

    // Duplicate check for location_id_external (excluding self)
    const duplicateCheck = await performQuery(
      `SELECT id FROM ${tables.org_location} WHERE location_id_external = ? AND id <> ? AND is_deleted = 2`,
      [location_id_external, id]
    );
    if (duplicateCheck && duplicateCheck.length > 0) {
      const errorMessage = "Location external ID already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check entity_id existence in legal entity table
    const isLegalEntity = await performQuery(
      `SELECT * FROM ${tables.org_legal_entity} WHERE id = ? AND is_deleted = 2`,
      [entity_id]
    );
    if (isLegalEntity.length === 0) {
      const errorMessage = "Legal Entity Not Found against entity_id";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    if(location_group_id){
      // Check location_group_id existence in location group table
      const isLocationGroup = await performQuery(
        `SELECT * FROM ${tables.org_location_group} WHERE id = ? AND is_deleted = 2`,
        [location_group_id]
      );
      if (isLocationGroup.length === 0) {
        const errorMessage = "Location Group Not Found against location_group_id";
        return sendErrorResponse(res, errorMessage, errorMessage);
      }
    }


    // System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Update Record
    await performQuery(`UPDATE ${tables.org_location} SET ? WHERE id = ?`, [{
      location_group_id: Number(location_group_id || oldRecord.location_group_id),
      location_name: location_name || oldRecord.location_name,
      effective_start_date: effective_start_date || oldRecord.effective_start_date,
      address_line_1: address_line_1 || oldRecord.address_line_1,
      address_line_2: address_line_2 || oldRecord.address_line_2,
      address_line_3: address_line_3 || oldRecord.address_line_3,
      locality: locality || oldRecord.locality,
      dependent_locality: dependent_locality || oldRecord.dependent_locality,
      administrative_area: administrative_area || oldRecord.administrative_area,
      sub_administrative_area: sub_administrative_area || oldRecord.sub_administrative_area,
      postal_code: postal_code || oldRecord.postal_code,
      country_code: country_code || oldRecord.country_code,
      sorting_code: sorting_code || oldRecord.sorting_code,
      language_code: language_code || oldRecord.language_code,
      status: status || oldRecord.status,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    }, id]);

    return sendResponse(res, {}, "Location updated successfully", 200);
  } catch (error) {
    console.error("Error While Updating Location: ", error);
    return sendErrorResponse(res, error, "Error while updating location");
  }
};

// DELETE Location (Soft Delete)
// module.exports.deleteLocation = async (req, res) => {
//   try {
//     const { id } = req.query;

//     if (!id) {
//       const errorMessage = "Location ID is required";
//       return sendErrorResponse(res, errorMessage, errorMessage);
//     }

//     const systemTime = await getSystemTime();
//     const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

//     await performQuery(`UPDATE ${tables.org_location} SET ? WHERE id = ?`, [
//       {
//         is_deleted: 1,
//         updated_at: currentTime,
//         updated_by: req?.user?.id,
//       },
//       id,
//     ]);

//     return sendResponse(res, {}, "Location deleted successfully", 200);
//   } catch (error) {
//     console.error("Error While Deleting Location: ", error);
//     return sendErrorResponse(res, error, "Error while deleting location");
//   }
// };
