const {
  sendErrorResponse,
  sendResponse,
} = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");


// CREATE Grade
module.exports.createGrade = async (req, res) => {
  try {
    const {
      grade_id_external,
      grade,
      category,
      effective_start_date,
      pay_range_max,
      pay_range_mid,
      pay_range_min,
      currency_code,
    } = req.body;

    if (!grade_id_external || !grade || !category || !effective_start_date) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check for duplicate grade_id_external
    const existingGrade = await performQuery(
      `SELECT id FROM ${tables.org_grade} WHERE (grade_id_external = ? OR grade = ?) AND is_deleted = 2`,
      [grade_id_external, grade]
    );
    if (existingGrade && existingGrade.length > 0) {
      const errorMessage = "Grade ID already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Get system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert record
    await performQuery(`INSERT INTO ${tables.org_grade} SET ?`, {
      grade_id_external,
      grade,
      category,
      effective_start_date,
      pay_range_max,
      pay_range_mid,
      pay_range_min,
      currency_code,
      created_at: currentTime,
      created_by: req?.user?.id,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    });

    // Send success response
    return sendResponse(res, {}, "Grade created successfully", 201);

  } catch (error) {
    console.error("Error while creating Grade:", error);
    return sendErrorResponse(res, error, "Error while creating Grade");
  }
};

// READ Grade(s)
module.exports.getGrade = async (req, res) => {
  try {
    const { id, grade_id_external, category } = req.query;

    let query = `
      SELECT * FROM ${tables.org_grade}
      WHERE is_deleted = 2
    `;

    if (id) query += ` AND id = ${id}`;
    if (grade_id_external) query += ` AND grade_id_external = '${grade_id_external}'`;
    if (category) query += ` AND category = ${category}`;

    query += ` ORDER BY id DESC`;

    const grades = await performQuery(query);
    return sendResponse(res, grades, "Grade(s) fetched successfully", 200);
  } catch (error) {
    console.error("Error while fetching Grade(s):", error);
    return sendErrorResponse(res, error, "Error while fetching Grade(s)");
  }
};

// UPDATE Grade
module.exports.updateGrade = async (req, res) => {
  try {
    const {
      id,
      grade_id_external,
      grade,
      category,
      effective_start_date,
      pay_range_min,
      pay_range_mid,
      pay_range_max,
      currency_code,
      status,
    } = req.body;

    if (!id || !grade_id_external || !grade || !category || !effective_start_date) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check if record exists
    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.org_grade} WHERE id = ? AND is_deleted = 2`,
      [id]
    );
    if (!existingRecord || existingRecord.length === 0) {
      const errorMessage = "Grade record not found";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    const oldRecord = existingRecord[0];

    // Check for duplicate grade_id_external
    const duplicateCheck = await performQuery(
      `SELECT id FROM ${tables.org_grade} WHERE (grade_id_external = ? OR grade = ?) AND id <> ? AND is_deleted = 2`,
      [grade_id_external, grade, id]
    );
    if (duplicateCheck && duplicateCheck.length > 0) {
      const errorMessage = "Grade ID already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Get system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Update record
    await performQuery(`UPDATE ${tables.org_grade} SET ? WHERE id = ?`, [{
      grade: grade || oldRecord.grade,
      category: category || oldRecord.category,
      effective_start_date: effective_start_date || oldRecord.effective_start_date,
      pay_range_min: pay_range_min || oldRecord.pay_range_min,
      pay_range_mid: pay_range_mid || oldRecord.pay_range_mid,
      pay_range_max: pay_range_max || oldRecord.pay_range_max,
      currency_code: currency_code || oldRecord.currency_code,
      status: status || oldRecord.status,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    }, id]);

    return sendResponse(res, {}, "Grade updated successfully", 200);
  } catch (error) {
    console.error("Error while updating Grade:", error);
    return sendErrorResponse(res, error, "Error while updating Grade");
  }
};

// DELETE Grade (Soft Delete)
// module.exports.deleteGrade = async (req, res) => {
//   try {
//     const { id } = req.query;

//     if (!id) {
//       const errorMessage = "Grade ID is required";
//       return sendErrorResponse(res, errorMessage, errorMessage);
//     }

//     // Get system time
//     const systemTime = await getSystemTime();
//     const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

//     // Soft delete
//     await performQuery(
//       `UPDATE ${tables.org_grade} SET ? WHERE id = ?`,
//       [
//         {
//           is_deleted: 1,
//           updated_at: currentTime,
//           updated_by: req?.user?.id,
//         },
//         id,
//       ]
//     );

//     return sendResponse(res, {}, "Grade deleted successfully", 200);
//   } catch (error) {
//     console.error("Error while deleting Grade:", error);
//     return sendErrorResponse(res, error, "Error while deleting Grade");
//   }
// };
