const { sendErrorResponse, sendResponse } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getTimezone, getSystemTime } = require("../../functions/getTimezone.js");

// Create Employee g lobal assignment
module.exports.createGlobalAssignment= async (req, res) => {
  try {
    const {
      assignment_id_external,
      employee_id_external ,
      start_date,
      end_date,
      status,
      type,
      reason
    } = req.body;

    // Required fields check
    if (
        !assignment_id_external || !employee_id_external || !start_date || !end_date || 
        !status || !type || !reason 
    ) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    

    // Duplicate Global-Assignment check
    const existing = await performQuery(
      `SELECT * FROM ${tables.emp_global_assignment} WHERE assignment_id_external = ? and is_deleted = 2`,
      [assignment_id_external]
    );
    if (existing.length > 0) {
      const errorMessage = "Global assignment record already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Duplicate employee_id_external check
    const existingEmployee = await performQuery(
      `SELECT * FROM ${tables.emp_global_assignment} WHERE employee_id_external = ? and is_deleted = 2`,
      [employee_id_external]
    );
    if (existingEmployee.length > 0) {
      const errorMessage = "Employee already has a global assignment";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Get system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert query
    await performQuery(`INSERT INTO ${tables.emp_global_assignment} SET ?`, {
      assignment_id_external,
      employee_id_external ,
      start_date,
      end_date,
      status,
      type,
      reason,
      created_at: currentTime,
      created_by: req?.user?.id,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    });

    return sendResponse(res, {}, "Global Assignment created successfully", 200);

  } catch (error) {
    console.error("Error while creating global assignment:", error);
    return sendErrorResponse(res, error, "Error while creating global assignment");
  }
};

// Read Employee global assignment
module.exports.getGlobalAssignment = async (req, res) => {
  try {
    const { id } = req.query;
    let query = `SELECT *FROM ${tables.emp_global_assignment} WHERE is_deleted = 2`;

    if (id) query += ` AND id = ${id}`;

    query += ` ORDER BY id DESC`;

    const result = await performQuery(query);

    return sendResponse(res, result, "Employee global assignment fetched successfully", 200);

  } catch (error) {
    console.error("Error While Fetching Employee global assignment ", error);
    return sendErrorResponse(res, error, "Error while fetching Employee global assignment");
  }
};

// Update Employee global assignment
module.exports.updateGlobalAssignment = async (req, res) => {
  try {
    const {
      id,
      start_date,
      end_date,
      status,
      type,
      reason
    } = req.body;

    // Required fields check
    if (!id ||!start_date ||!end_date ||!status ||!type ||!reason ) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    
    // Check Record Exists
    const existing = await performQuery(
      `SELECT * FROM ${tables.emp_global_assignment} WHERE id = ? AND is_deleted = 2`,
      [id]
    );

    if (!existing || existing.length === 0) {
      const errorMessage = "Global assignment record not found";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    const oldRecord = existing[0];


    // Duplicate Global assignment_id_external check
    const duplicate = await performQuery(
      `SELECT * FROM ${tables.emp_global_assignment} WHERE assignment_id_external = ? and id<>? and is_deleted = 2`,
      [oldRecord.assignment_id_external, id]
    );
    if (duplicate.length > 0) {
      const errorMessage = "Global Assignment record already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Duplicate employee_id_external check
    const duplicateEmployee = await performQuery(
      `SELECT * FROM ${tables.emp_global_assignment} WHERE employee_id_external = ? and id<>? and is_deleted = 2`,
      [oldRecord.employee_id_external, id]
    );
    if (duplicateEmployee.length > 0) {
      const errorMessage = "Employee record already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    
    // System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Prepare update object
    const updatedData = {
      start_date : start_date || oldRecord.start_date,
      end_date : end_date || oldRecord.end_date,
      status : status || oldRecord.status,
      type : type || oldRecord.type,
      reason : reason || oldRecord.reason,
      updated_at: currentTime,
      updated_by: req?.user?.id || null,
    };

    // Update query
    await performQuery(`UPDATE ${tables.emp_global_assignment} SET ? WHERE id = ?`, [
      updatedData,
      id,
    ]);

    return sendResponse(res, {}, "Global Assignment updated successfully", 200);

  } catch (error) {
    console.error("Error While Updating global assignment: ", error);
    return sendErrorResponse(res, error, "Error while updating global assignment");
  }
};