const { sendErrorResponse, sendResponse } = require("../../utils");
const { performQuery } = require("../../utils/db");
const { tables } = require("../../utils/tables");
const { getSystemTime } = require("../../functions/getTimezone");
const moment = require("moment-timezone");

// API to add email ID for a person --> /personRecord/email POST API
module.exports.addEmailId = async (req, res) => {
    try {
        // Get Variables from Request Body
        const { 
            person_id,
            email,
            email_type,
            is_primary
        } = req.body;

        // Validate Required Fields
        if (!person_id || !email || !email_type || !is_primary) {
            return sendErrorResponse(res, null, "Missing required fields: person_id, email, email_type, is_primary");
        }

        // Validate Email Format
        const regex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        if (!regex.test(email)) {
            return sendErrorResponse(res, null, "Invalid email format");
        }
        
        // check if person exists
        const personRecord = await performQuery(
            `SELECT * FROM ${tables.per_person} WHERE id = ? AND is_deleted = 2`,
            [person_id]
        );
        if (personRecord.length === 0) {
            return sendErrorResponse(res, null, "Person not found");
        }

        // Check if same email already exists in records
        const existingEmail = await performQuery(
            `SELECT * FROM ${tables.per_email} WHERE email = ? AND is_deleted = 2`,
            [email]
        );
        if (existingEmail.length > 0) {
            return sendErrorResponse(res, null, "Email already exists");
        }
            
        // Get System Time
        const systemTime = await getSystemTime();
        const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

        // Insert Email Record
        const insertResult = await performQuery(
            `INSERT INTO ${tables.per_email} SET ?`,
            {
                person_id: person_id,
                email: email,
                email_type: email_type,
                is_primary: is_primary,
                created_by: req?.user?.id,
                created_at: currentTime,
                updated_by: req?.user?.id,
                updated_at: currentTime
            }
        );

        if(is_primary == 1){
            const updateOtherEmails = await performQuery(
                `UPDATE ${tables.per_email} SET ? WHERE person_id = ? AND id <> ? AND is_deleted = 2`,
                [
                    {
                    is_primary: 2,
                    updated_by: req?.user?.id,
                    updated_at: currentTime
                    }, 
                    person_id, 
                    insertResult.insertId
                ]
            );
        }

        // Return Success Response
        return sendResponse(res, [], "Email added successfully", 201);

    } catch (error) {
        return sendErrorResponse(res, error, "Error while adding email");
    }
}

// API to get email IDs --> /personRecord/email GET API
module.exports.getEmailIds = async (req, res) => {
    try {
        // Get person_id from Query Params
        const { person_id, person_id_external } = req.query;

        // Query
        let query = `
            Select 
                pe.*, 
                per.display_name AS person_name
            From ${tables.per_email} pe
            Inner Join ${tables.per_person} per ON pe.person_id = per.id
            Where pe.is_deleted = 2
        `;

        // Apply Filters
        if (person_id) {
            query += ` AND pe.person_id = ${person_id} `;
        }
        else if (person_id_external) {
            query += ` AND per.person_id_external = '${person_id_external}' `;
        }

        console.log(query);
        

        // Fetch Email IDs
        const emailRecords = await performQuery(query);

        // Return Success Response
        return sendResponse(res, emailRecords, "Email IDs fetched successfully");

    } catch (error) {
        return sendErrorResponse(res, error, "Error while fetching email IDs");
    }
}