const { sendErrorResponse, sendResponse } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");
const { error } = require("winston");


// CREATE Non Recurring Pay Component
module.exports.createNonRecurringPayComp = async (req, res) => {
  try {
    const {
      non_recurring_pay_comp_id_external,
      value,
      currency_code,
      pay_date,
      cost_center_id,
      no_of_units,
      units_of_measure,
      reference_id,
      start_date,
      end_date,
      employee_id,
      note,
    } = req.body;

    // Validation
    if (
      !non_recurring_pay_comp_id_external || !value || !currency_code || !pay_date || 
      !cost_center_id || !no_of_units || !units_of_measure || !reference_id || 
      !start_date || !end_date || !employee_id
    ) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Validate employee existence
    // const employeeExists = await performQuery(
    //   `SELECT * FROM ${tables.per_person} WHERE id = ? AND is_deleted = 2`,
    //   [employee_id]
    // );

    // if (employeeExists.length === 0) {
    //   const errorMessage = "Employee not found against employee_id";
    //   return sendErrorResponse(res, errorMessage, errorMessage);
    // }

    // Validate cost center id existence
    const costCenterIDExists = await performQuery(
      `SELECT * FROM ${tables.org_cost_center} WHERE id = ? AND is_deleted = 2`,
      [cost_center_id]
    );
    if (costCenterIDExists.length === 0) {
      const errorMessage = "Cost Center not found against cost_center_id";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check for duplicate external ID
    const duplicateCheck = await performQuery(
      `SELECT id FROM ${tables.org_non_recurring_pay_component} WHERE non_recurring_pay_comp_id_external = ? AND is_deleted = 2`,
      [non_recurring_pay_comp_id_external]
    );
    if (duplicateCheck && duplicateCheck.length > 0) {
      const errorMessage = "Non Recurring Pay Component external ID already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert record
    await performQuery(`INSERT INTO ${tables.org_non_recurring_pay_component} SET ?`, {
      non_recurring_pay_comp_id_external,
      value,
      currency_code,
      pay_date,
      cost_center_id,
      no_of_units,
      units_of_measure,
      reference_id,
      start_date,
      end_date,
      employee_id,
      note,
      created_at: currentTime,
      created_by: req?.user?.id,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    });

    // Success Response
    return sendResponse(res, {}, "Non Recurring Pay Component created successfully", 201);

  } catch (error) {
    console.error("Error While Creating Non Recurring Pay Component: ", error);
    return sendErrorResponse(res, error, "Error while creating Non Recurring Pay Component");
  }
};


// READ Non Recurring Pay Component(s)
module.exports.getNonRecurringPayComp = async (req, res) => {
  try {
    const { id } = req.query;

    let query = `
      SELECT *
      FROM ${tables.org_non_recurring_pay_component}
      WHERE is_deleted = 2
    `;

    if (id) query += ` AND id = ${id}`;
    query += ` ORDER BY id DESC`;

    const result = await performQuery(query);

    return sendResponse(res, result, "Non Recurring Pay Component(s) fetched successfully", 200);

  } catch (error) {
    console.error("Error While Fetching Non Recurring Pay Component(s): ", error);
    return sendErrorResponse(res, error, "Error while fetching Non Recurring Pay Component(s)");
  }
};


// UPDATE Non Recurring Pay Component (Versioned Update)
module.exports.updateNonRecurringPayComp = async (req, res) => {
  try {
    const { 
      id,
      non_recurring_pay_comp_id_external,
      value,
      currency_code,
      pay_date,
      cost_center_id,
      no_of_units,
      units_of_measure,
      reference_id,
      start_date,
      end_date,
      employee_id,
      note,
    } = req.body;

    if (
      !id || !non_recurring_pay_comp_id_external || !value || !currency_code || !pay_date || 
      !cost_center_id || !no_of_units || !units_of_measure || !reference_id || 
      !start_date || !end_date || !employee_id
    ) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.org_non_recurring_pay_component} WHERE id = ? AND is_deleted = 2`,
      [id]
    );

    if (!existingRecord || existingRecord.length === 0) {
      const errorMessage = "Non Recurring Pay Component not found";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    const oldRecord = existingRecord[0];

    

    // Validate employee existence
    // const employeeExists = await performQuery(
    //   `SELECT * FROM ${tables.per_person} WHERE id = ? AND is_deleted = 2`,
    //   [employee_id]
    // );

    // if (employeeExists.length === 0) {
    //   const errorMessage = "Employee not found against employee_id";
    //   return sendErrorResponse(res, error, "Employee not found against employee_id");
    // }

    // Validate cost center id existence
    const costCenterIDExists = await performQuery(
      `SELECT * FROM ${tables.org_cost_center} WHERE id = ? AND is_deleted = 2`,
      [cost_center_id]
    );
    if (costCenterIDExists.length === 0) {
      const errorMessage = "Cost Center not found against cost_center_id";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Duplicate check (excluding current)
    const duplicateCheck = await performQuery(
      `SELECT id FROM ${tables.org_non_recurring_pay_component} WHERE non_recurring_pay_comp_id_external = ? AND id <> ? AND is_deleted = 2`,
      [non_recurring_pay_comp_id_external, id]
    );

    if (duplicateCheck && duplicateCheck.length > 0) {
      const errorMessage = "Non Recurring Pay Component external ID already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Update Record
    await performQuery(`UPDATE ${tables.org_non_recurring_pay_component} SET ? WHERE id = ?`, [{
      value: value || oldRecord.value,
      currency_code: currency_code || oldRecord.currency_code,
      pay_date: pay_date || oldRecord.pay_date,
      cost_center_id: cost_center_id || oldRecord.cost_center_id,
      no_of_units: no_of_units || oldRecord.no_of_units,
      units_of_measure: units_of_measure || oldRecord.units_of_measure,
      reference_id: reference_id || oldRecord.reference_id,
      start_date: start_date || oldRecord.start_date,
      end_date: end_date || oldRecord.end_date,
      employee_id: employee_id || oldRecord.employee_id,
      note: note || oldRecord.note,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    }, id]);

    return sendResponse(res, {}, "Non Recurring Pay Component updated successfully", 200);
    
  } catch (error) {
    console.error("Error While Updating Non Recurring Pay Component: ", error);
    return sendErrorResponse(res, error, "Error while updating Non Recurring Pay Component");
  }
};


// DELETE Non Recurring Pay Component (Soft Delete)
// module.exports.deleteNonRecurringPayComp = async (req, res) => {
//   try {
//     const { id } = req.query;

//     if (!id) {
//       const errorMessage = "Non Recurring Pay Component ID is required";
//       return sendErrorResponse(res, errorMessage, errorMessage);
//     }

//     const systemTime = await getSystemTime();
//     const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

//     await performQuery(`UPDATE ${tables.org_non_recurring_pay_component} SET ? WHERE id = ?`, [
//       {
//         is_deleted: 1,
//         updated_at: currentTime,
//         updated_by: req?.user?.id,
//       },
//       id,
//     ]);

//     return sendResponse(res, {}, "Non Recurring Pay Component deleted successfully", 200);
//   } catch (error) {
//     console.error("Error While Deleting Non Recurring Pay Component: ", error);
//     return sendErrorResponse(res, error, "Error while deleting Non Recurring Pay Component");
//   }
// };
