const {
  sendErrorResponse,
  sendResponse,
} = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");


// CREATE Job
module.exports.createJob = async (req, res) => {
  try {
    const {
      job_id_external,
      name_us,
      name_sa,
      pay_group_id,
      effective_start_date,
    } = req.body;

    if (!job_id_external || !pay_group_id || !name_us || !effective_start_date) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check if Pay Group exists
    const payGroupExists = await performQuery(
      `SELECT * FROM ${tables.org_pay_group} WHERE id = ? AND is_deleted = 2`,
      [pay_group_id]
    );
    if (payGroupExists.length === 0) {
      const errorMessage = "Pay Group ID not found in Pay Group table";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check for duplicate job_id_external, name_us
    const existingJob = await performQuery(
      `SELECT id FROM ${tables.org_job_classification} WHERE (job_id_external = ? OR name_us = ?) AND is_deleted = 2`,
      [job_id_external, name_us]
    );
    if (existingJob && existingJob.length > 0) {
      const errorMessage = "Job ID already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Get system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert Record
    await performQuery(`INSERT INTO ${tables.org_job_classification} SET ?`, {
      job_id_external,
      name_us,
      name_sa,
      pay_group_id,
      effective_start_date,
      created_at: currentTime,
      created_by: req?.user?.id,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    });

    return sendResponse(res, {}, "Job created successfully", 201);
  } catch (error) {
    console.error("Error while creating Job:", error);
    return sendErrorResponse(res, error, "Error while creating Job");
  }
};


// READ Job(s)
module.exports.getJob = async (req, res) => {
  try {
    const { id, job_id_external, pay_group_id } = req.query;

    let query = `
      SELECT 
        job.*, 
        paygroup.pay_group_id_external, paygroup.group_name
      FROM ${tables.org_job_classification} AS job
      LEFT JOIN ${tables.org_pay_group} AS paygroup
        ON job.pay_group_id = paygroup.id AND paygroup.is_deleted = 2
      WHERE job.is_deleted = 2
    `;

    if (id) query += ` AND job.id = ${id}`;
    if (job_id_external) query += ` AND job.job_id_external = '${job_id_external}'`;
    if (pay_group_id) query += ` AND job.pay_group_id = ${pay_group_id}`;

    query += ` ORDER BY job.id DESC`;

    const jobs = await performQuery(query);
    return sendResponse(res, jobs, "Job(s) fetched successfully", 200);
  } catch (error) {
    console.error("Error while fetching Job(s):", error);
    return sendErrorResponse(res, error, "Error while fetching Job(s)");
  }
};


// UPDATE Job
module.exports.updateJob = async (req, res) => {
  try {
    const {
      id,
      job_id_external,
      name_us,
      name_sa,
      pay_group_id,
      effective_start_date,
      status,
    } = req.body;

    if (!id || !job_id_external || !name_us || !pay_group_id || !effective_start_date) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check if Pay Group exists
    const payGroupExists = await performQuery(
      `SELECT * FROM ${tables.org_pay_group} WHERE id = ? AND is_deleted = 2`,
      [pay_group_id]
    );
    if (payGroupExists.length === 0) {
      const errorMessage = "Pay Group ID not found in Pay Group table";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check if record exists
    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.org_job_classification} WHERE id = ? AND is_deleted = 2`,
      [id]
    );
    if (!existingRecord || existingRecord.length === 0) {
      const errorMessage = "Job record not found";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    const oldRecord = existingRecord[0];

    // Check for duplicate job_id_external
    const duplicateCheck = await performQuery(
      `SELECT id FROM ${tables.org_job_classification} WHERE (job_id_external = ? OR name_us = ?) AND id <> ? AND is_deleted = 2`,
      [job_id_external, name_us, id]
    );
    if (duplicateCheck && duplicateCheck.length > 0) {
      const errorMessage = "Job ID already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Get system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Update record
    await performQuery(`UPDATE ${tables.org_job_classification} SET ? WHERE id = ?`, [{
      name_us: name_us || oldRecord.name_us,
      name_sa: name_sa || oldRecord.name_sa,
      pay_group_id: pay_group_id || oldRecord.pay_group_id,
      effective_start_date: effective_start_date || oldRecord.effective_start_date,
      status: status || oldRecord.status,
      is_deleted: 2,
      created_at: currentTime,
      created_by: req?.user?.id,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    }, id]);

    return sendResponse(res, {}, "Job updated successfully", 200);
  } catch (error) {
    console.error("Error while updating Job:", error);
    return sendErrorResponse(res, error, "Error while updating Job");
  }
};


// DELETE Job (Soft Delete)
// module.exports.deleteJob = async (req, res) => {
//   try {
//     const { id } = req.query;

//     if (!id) {
//       const errorMessage = "Job ID is required";
//       return sendErrorResponse(res, errorMessage, errorMessage);
//     }

//     // Get system time
//     const systemTime = await getSystemTime();
//     const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

//     // Soft delete
//     await performQuery(
//       `UPDATE ${tables.org_job_classification} SET ? WHERE id = ?`,
//       [
//         {
//           is_deleted: 1,
//           updated_at: currentTime,
//           updated_by: req?.user?.id,
//         },
//         id,
//       ]
//     );

//     return sendResponse(res, {}, "Job deleted successfully", 200);
//   } catch (error) {
//     console.error("Error while deleting Job:", error);
//     return sendErrorResponse(res, error, "Error while deleting Job");
//   }
// };
