const {
  sendErrorResponse,
  sendResponse,
} = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");

// CREATE Contact
module.exports.createContact = async (req, res) => {
  try {
    const {
      person_id,
      country_code,
      contact_no,
      contact_type = 1,
      is_primary = 2,
    } = req.body;

    if (!person_id || !country_code || !contact_no || !contact_type || !is_primary) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check if person exists
    const personExists = await performQuery(
      `SELECT id FROM ${tables.per_person} WHERE id = ? AND is_deleted = 2`,
      [person_id]
    );
    if (!personExists || personExists.length === 0) {
      const errorMessage = "Person not found";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check for duplicate contact in record
    const existingContact = await performQuery(
      `SELECT id FROM ${tables.per_contact_no} WHERE contact_no = ? AND is_deleted = 2`,
      [person_id, contact_no]
    );
    if (existingContact && existingContact.length > 0) {
      const errorMessage = "This contact number already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Get system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert into Database
    const insertRecord = await performQuery(`INSERT INTO ${tables.per_contact_no} SET ?`, {
      person_id,
      country_code,
      contact_no,
      contact_type,
      is_primary,
      created_at: currentTime,
      created_by: req?.user?.id,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    });

    // If Current is Primary, make others non-primary (isPrimary = 2)
    if(is_primary == 1){
      await performQuery(
        `UPDATE ${tables.per_contact_no} SET ? WHERE id <> ? AND person_id = ?`,
        [
            {
                is_primary: 2,
                updated_at: currentTime,
                updated_by: req?.user?.id,
            },
            insertRecord.insertId, person_id
        ]
      );
    }

    // Success response
    return sendResponse(res, {}, "Contact created successfully", 201);

  } catch (error) {
    console.error("Error While Creating Contact: ", error);
    return sendErrorResponse(res, error, "Error while creating contact");
  }
};

// READ Contact(s)
module.exports.getContact = async (req, res) => {
  try {
    const { person_id, person_id_external } = req.query;

    let query = `
      SELECT 
        con.*,
        per.display_name AS person_name, per.person_id_external
      FROM ${tables.per_contact_no} AS con
      LEFT JOIN ${tables.per_person} AS per ON con.person_id = per.id
      WHERE con.is_deleted = 2
    `;

    if (person_id) query += ` AND con.person_id = ${person_id}`;
    if (person_id_external) query += ` AND per.person_id_external = '${person_id_external}'`;

    // Sort by most recent id
    query += ` ORDER BY con.id DESC`;

    const contacts = await performQuery(query);
    return sendResponse(res, contacts, "Contact(s) fetched successfully", 200);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while fetching contact(s)");
  }
};

// UPDATE Contact
module.exports.updateContact = async (req, res) => {
  try {
    const {
      id,
      country_code,
      contact_no,
      contact_type,
      is_primary,
    } = req.body;

    if (!id || !country_code || !contact_no || !is_primary) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check if contact exists
    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.per_contact_no} WHERE id = ? AND is_deleted = 2`,
      [id]
    );
    if (!existingRecord || existingRecord.length === 0) {
      const errorMessage = "Contact not found";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    const oldRecord = existingRecord[0];

    // Check for duplicate contact_no in record
    const duplicateCheck = await performQuery(
      `SELECT id FROM ${tables.per_contact_no} WHERE contact_no = ? AND id <> ? AND is_deleted = 2`,
      [contact_no, id]
    );
    if (duplicateCheck && duplicateCheck.length > 0) {
      const errorMessage = "Contact number already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Mark existing record as deleted (to maintain history)
    await performQuery(
      `UPDATE ${tables.per_contact_no} SET ? WHERE id = ?`,
      [
        {
          is_deleted: 1,
          updated_at: currentTime,
          updated_by: req?.user?.id,
        },
        id,
      ]
    );

    // Insert new updated record
    const insertRecord = await performQuery(`INSERT INTO ${tables.per_contact_no} SET ?`, {
      person_id: oldRecord.person_id,
      country_code: country_code || oldRecord.country_code,
      contact_no: contact_no || oldRecord.contact_no,
      contact_type: contact_type || oldRecord.contact_type,
      is_primary: is_primary || oldRecord.is_primary,
      is_deleted: 2,
      created_at: currentTime,
      created_by: req?.user?.id,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    });

    if(is_primary){
        // is this contact is primary mark other as non primary
        await performQuery(
        `UPDATE ${tables.per_contact_no} SET ? WHERE id <> ? AND person_id = ?`,
        [
            {
                is_primary: 2,
                updated_at: currentTime,
                updated_by: req?.user?.id,
            },
            insertRecord.insertId, oldRecord.person_id
        ]
        );
    }


    return sendResponse(res, {}, "Contact updated successfully", 200);
  } catch (error) {
    return sendErrorResponse(res, error, "Error while updating contact");
  }
};

// DELETE Contact (Soft Delete)
module.exports.deleteContact = async (req, res) => {
  try {
    const { id } = req.query;

    if (!id) {
      const errorMessage = "Contact ID is required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Soft delete
    await performQuery(
      `UPDATE ${tables.per_contact_no} SET ? WHERE id = ?`,
      [
        {
          is_deleted: 1,
          updated_at: currentTime,
          updated_by: req?.user?.id,
        },
        id,
      ]
    );

    return sendResponse(res, {}, "Contact deleted successfully", 200);
  } catch (error) {
    return sendErrorResponse(res, error, "Error while deleting contact");
  }
};
