const { sendErrorResponse, sendResponse } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getTimezone, getSystemTime } = require("../../functions/getTimezone.js");

// Create Employee
module.exports.createTermination= async (req, res) => {
  const uploadedFiles = [];
  try {
    const {
      employee_id_external,
      type,
      request_date,
      final_working_day,
      remarks,
      approved_by,
      approval_at,
    } = req.body;

    // Required fields check
    if (!employee_id_external || !type || !request_date || !final_working_day) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    
    // File upload
    let baseURL = `${req.protocol}://${req.get("host")}/uploads`;
    const resignation_doc = req.files?.resignation_doc? req.files.resignation_doc[0].filename: null;
    const termination_doc = req.files?.termination_doc? req.files.termination_doc[0].filename: null;
    
    if (resignation_doc) uploadedFiles.push(resignation_doc);
    if (termination_doc) uploadedFiles.push(termination_doc);
    const resignation_doc_URL = resignation_doc? `${baseURL}/${resignation_doc}`: null;
    const termination_doc_URL = termination_doc? `${baseURL}/${termination_doc}`: null;

    // Duplicate position check
    const existing = await performQuery(
      `SELECT * FROM ${tables.emp_termination} WHERE employee_id_external = ? and is_deleted = 2`,
      [employee_id_external]
    );
    if (existing.length > 0) {
      const errorMessage = "Termination record already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Get system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert query
    await performQuery(`INSERT INTO ${tables.emp_termination} SET ?`, {
      employee_id_external,
      type,
      request_date,
      final_working_day,
      remarks,
      approved_by,
      approval_at,
      resignation_doc: resignation_doc_URL,
      termination_doc: termination_doc_URL,      
      created_at: currentTime,
      created_by: req?.user?.id,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    });

    return sendResponse(res, {}, "Termination created successfully", 200);

  } catch (error) {
    console.error("Error while creating termination:", error);
    return sendErrorResponse(res, error, "Error while creating termination");
  }
};

// Read Employee
module.exports.getTermination = async (req, res) => {
  try {
    const { id } = req.query;
    let query = `SELECT *FROM ${tables.emp_termination} WHERE is_deleted = 2`;

    if (id) query += ` AND id = ${id}`;

    query += ` ORDER BY id DESC`;

    const result = await performQuery(query);

    return sendResponse(res, result, "Termination fetched successfully", 200);

  } catch (error) {
    console.error("Error While Fetching termination: ", error);
    return sendErrorResponse(res, error, "Error while fetching termination");
  }
};

// Update Employee
module.exports.updateTermination = async (req, res) => {
  const uploadedFiles = [];
  try {
    const {
      id,
      type,
      request_date,
      final_working_day,
      remarks,
      approved_by,
      approval_at,
    } = req.body;

    // Required fields check
    if (!id ||!type ||!request_date ||!final_working_day) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    
    // File upload
    let baseURL = `${req.protocol}://${req.get("host")}/uploads`;
    const resignation_doc = req.files?.resignation_doc? req.files.resignation_doc[0].filename: null;
    const termination_doc = req.files?.termination_doc? req.files.termination_doc[0].filename: null;
    
    if (resignation_doc) uploadedFiles.push(resignation_doc);
    if (termination_doc) uploadedFiles.push(termination_doc);
    const resignation_doc_URL = resignation_doc? `${baseURL}/${resignation_doc}`: null;
    const termination_doc_URL = termination_doc? `${baseURL}/${termination_doc}`: null;

    // Check Record Exists
    const existing = await performQuery(
      `SELECT * FROM ${tables.emp_termination} WHERE id = ? AND is_deleted = 2`,
      [id]
    );

    if (!existing || existing.length === 0) {
      const errorMessage = "Termination record not found";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    const oldRecord = existing[0];


    // Duplicate position check
    const duplicate = await performQuery(
      `SELECT * FROM ${tables.emp_termination} WHERE employee_id_external = ? and id<>? and is_deleted = 2`,
      [oldRecord.employee_id_external, id]
    );
    if (duplicate.length > 0) {
      const errorMessage = "Termination record already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Prepare update object
    const updatedData = {
      type : type || oldRecord.type,
      request_date : request_date || oldRecord.request_date,
      final_working_day : final_working_day || oldRecord.final_working_day,
      remarks : remarks || oldRecord.remarks,
      approved_by : approved_by || oldRecord.approved_by,
      approval_at : approval_at || oldRecord.approval_at,
      resignation_doc: resignation_doc ? `${baseURL}/${resignation_doc}`: oldRecord.resignation_doc,
      termination_doc: termination_doc ? `${baseURL}/${termination_doc}`: oldRecord.termination_doc,
      updated_at: currentTime,
      updated_by: req?.user?.id || null,
    };

    // Update query
    await performQuery(`UPDATE ${tables.emp_termination} SET ? WHERE id = ?`, [
      updatedData,
      id,
    ]);

    return sendResponse(res, {}, "Termination updated successfully", 200);

  } catch (error) {
    console.error("Error While Updating termination: ", error);
    return sendErrorResponse(res, error, "Error while updating termination");
  }
};