const {
  sendErrorResponse,
  sendResponse,
} = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");
const { isUndefined } = require("lodash");
const fs = require('fs');
const path = require('path');


// CREATE Passport
module.exports.createPassport = async (req, res) => {
  const uploadedFiles = [];
  try {
    const {
      person_id,
      country,
      passport_no,
      issue_date,
      issue_date_hijri,
      issue_place,
      issuing_authority,
      validated,
      expiry_date,
      expiry_date_hijri,
      notes
    } = req.body;

    if (
      !person_id || !country || !passport_no || !issue_date || !issue_place || !issuing_authority ||
      !validated || !expiry_date
    ) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
     // Handle file uploads
    let baseURL = `${req.protocol}://${req.get("host")}/uploads`;
    const attachment_1 = req.files?.attachment_1 ? req.files.attachment_1[0].filename : null;
    const attachment_2 = req.files?.attachment_2 ? req.files.attachment_2[0].filename : null;
    if (attachment_1) uploadedFiles.push(attachment_1);
    if (attachment_2) uploadedFiles.push(attachment_2);

    // return error if either file is missing
    if (!attachment_1 && !attachment_2) {
      return sendErrorResponse(res, "Attach Picture of Passport", "Missing required Files", 400);
    }

    // Check if person exists
    const personExists = await performQuery(
      `SELECT id FROM ${tables.per_person} WHERE id = ? AND is_deleted = 2`,
      [person_id]
    );
    if (!personExists || personExists.length === 0) {
      const errorMessage = "Person not found";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check for existing records against person_id / passport_no
    const existingRecords = await performQuery(
      `SELECT id FROM ${tables.per_passport} WHERE (person_id = ? OR passport_no = ?) AND is_deleted = 2`,
      [person_id, passport_no]
    );
    if (existingRecords && existingRecords.length > 0) {
      const errorMessage = "Passport already exists for this person or passport number";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    
    // Calculate completion and is_completed
    let total_fields = 13; // Total fields to consider for completion
    let filled_fields = 0;

    if (person_id) filled_fields++;
    if (country) filled_fields++;
    if (passport_no) filled_fields++;
    if (issue_date) filled_fields++;
    if (issue_date_hijri) filled_fields++;
    if (issue_place) filled_fields++;
    if (issuing_authority) filled_fields++;
    if (validated)filled_fields++;
    if (expiry_date) filled_fields++;
    if (expiry_date_hijri) filled_fields++;
    if (attachment_1) filled_fields++;
    if (attachment_2) filled_fields++;
    if (notes)filled_fields++;

    const completion = Math.round((filled_fields / total_fields) * 100);
    const is_completed = completion === 100 ? 2 : 1; // 2 for completed, 1 for not completed

    // Get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");
                        
    // create File URLs
    const attachment_1_URL = attachment_1 ? `${baseURL}/${attachment_1}` : null;
    const attachment_2_URL = attachment_2 ? `${baseURL}/${attachment_2}` : null;

    // Insert into Database
    const newPassport = await performQuery(
      `INSERT INTO ${tables.per_passport} SET ?`, {
        person_id: person_id,
        country: country,
        passport_no: passport_no,
        issue_date: issue_date,
        issue_date_hijri: issue_date_hijri,
        issue_place: issue_place,
        issuing_authority: issuing_authority,
        validated: validated,
        expiry_date: expiry_date,
        expiry_date_hijri: expiry_date_hijri,
        attachment_1: attachment_1_URL,
        attachment_2: attachment_2_URL,
        notes: notes,
        completion: completion,
        is_completed: is_completed,
        created_at: currentTime,
        created_by: req?.user?.id,
        updated_at: currentTime,
        updated_by: req?.user?.id
      }
    );
    
    // Send success response
    return sendResponse(res, {}, "Passport created successfully", 201);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while creating passport");
  }
};


// READ Passport(s)
module.exports.getPassport = async (req, res) => {
  try {
    const { person_id, person_id_external } = req.query;

    let query = `
      SELECT 
        pass.*,
        per.display_name AS person_name, per.person_id_external,
        cre.display_name AS created_by_name,
        upd.display_name AS updated_by_name 
      FROM ${tables.per_passport} AS pass
      Left JOIN ${tables.per_person} AS per ON pass.person_id = per.id
      Left JOIN ${tables.per_person} AS cre ON pass.created_by = cre.id
      Left JOIN ${tables.per_person} AS upd ON pass.updated_by = upd.id 
      WHERE pass.is_deleted = 2`;

    if (person_id) {
      query += ` AND pass.person_id = ${person_id}`;
    }
    if (person_id_external) {
      query += ` AND per.person_id_external = '${person_id_external}'`;
    }

    // sort by most recent id
    query += ` ORDER BY pass.id DESC`;

    // Execute Query
    const passports = await performQuery(query);

    // Send Response
    return sendResponse(res, passports, "Passport(s) fetched successfully", 200);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while fetching passport");
  }
};


// UPDATE Passport
module.exports.updatePassport = async (req, res) => {
  const uploadedFiles = [];
  try {
    const {
      id,
      country,
      passport_no,
      issue_date,
      issue_date_hijri,
      issue_place,
      issuing_authority,
      validated,
      expiry_date,
      expiry_date_hijri,
      notes
    } = req.body;

    if (
      !id || !passport_no || !issue_date || !expiry_date || !country || !issue_place || !issuing_authority || !validated
    ) {
      const errorMessage = "required fields are Missing";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Handle file uploads
    let baseURL = `${req.protocol}://${req.get("host")}/uploads`;
    const attachment_1 = req.files?.attachment_1 ? req.files.attachment_1[0].filename : null;
    const attachment_2 = req.files?.attachment_2 ? req.files.attachment_2[0].filename : null;
    if (attachment_1) uploadedFiles.push(attachment_1);
    if (attachment_2) uploadedFiles.push(attachment_2);

    // check if passport_no exist for other records
    const existingPassportNoRecords = await performQuery(
      `SELECT id FROM ${tables.per_passport} WHERE passport_no = ? AND id != ? AND is_deleted = 2`,
      [passport_no, id]
    );
    if (existingPassportNoRecords && existingPassportNoRecords.length > 0) {
      const errorMessage = "Passport number already exists for another record";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // get Previous Record
    const existingRecord = await performQuery(`
      SELECT * FROM ${tables.per_passport} WHERE id = ?`, [id]
    );
    if (!existingRecord || existingRecord.length === 0) {
      const errorMessage = "Passport not found";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    const previousRecord = existingRecord[0];

    // Calculate completion and is_completed
    let total_fields = 13; // Total fields to consider for completion
    let filled_fields = 0;

    if (previousRecord.person_id) filled_fields++;
    if (country || previousRecord.country) filled_fields++;
    if (passport_no || previousRecord.passport_no) filled_fields++;
    if (issue_date || previousRecord.issue_date) filled_fields++;
    if (issue_date_hijri || previousRecord.issue_date_hijri) filled_fields++;
    if (issue_place || previousRecord.issue_place) filled_fields++;
    if (issuing_authority || previousRecord.issuing_authority) filled_fields++;
    if (validated || previousRecord.validated) filled_fields++;
    if (expiry_date || previousRecord.expiry_date) filled_fields++;
    if (expiry_date_hijri || previousRecord.expiry_date_hijri) filled_fields++;
    if (attachment_1 || previousRecord.attachment_1) filled_fields++;
    if (attachment_2 || previousRecord.attachment_2) filled_fields++;
    if (notes || previousRecord.notes) filled_fields++;

    const completion = Math.round((filled_fields / total_fields) * 100);
    const is_completed = completion === 100 ? 2 : 1; // 2 for completed, 1 for not completed

    // Get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");
                        
    // create File URLs
    const attachment_1_URL = attachment_1 ? `${baseURL}/${attachment_1}` : null;
    const attachment_2_URL = attachment_2 ? `${baseURL}/${attachment_2}` : null;

    // Mark existing Record as deleted to maintain history
    const updateExistingRecord = await performQuery(`
      UPDATE ${tables.per_passport} SET ? WHERE id = ?`, [
      {
        is_deleted: 1,
        updated_at: currentTime,
        updated_by: req?.user?.id,
      }, id]
    );
    
    // create New Record with updated data
    const newRecord = await performQuery(
      `INSERT INTO ${tables.per_passport} SET ?`, {
        person_id: previousRecord.person_id,
        country: country || previousRecord.country,
        passport_no: passport_no || previousRecord.passport_no,
        issue_date: issue_date || previousRecord.issue_date,
        issue_date_hijri: issue_date_hijri || previousRecord.issue_date_hijri,
        issue_place: issue_place || previousRecord.issue_place,
        issuing_authority: issuing_authority || previousRecord.issuing_authority,
        validated: validated || previousRecord.validated,
        expiry_date: expiry_date || previousRecord.expiry_date,
        expiry_date_hijri: expiry_date_hijri || previousRecord.expiry_date_hijri,
        attachment_1: attachment_1_URL || previousRecord.attachment_1,
        attachment_2: attachment_2_URL || previousRecord.attachment_2,
        notes: notes || previousRecord.notes,
        completion: completion,
        is_completed: is_completed,
        created_at: currentTime,  
        created_by: req?.user?.id,
        updated_at: currentTime,
        updated_by: req?.user?.id
      }
    );

    // Send success response
    return sendResponse(res, {}, "Passport updated successfully", 200);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while updating passport");
  }
};


// DELETE Passport (Soft Delete)
module.exports.deletePassport = async (req, res) => {
  try {
    const { id } = req.query;

    if (isUndefined(id)) {
      const errorMessage = "Passport ID is required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Get System Time
        const systemTime = await getSystemTime();
        const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    const deleteQuery = `
      UPDATE ${tables.per_passport}
      SET ? WHERE id = ?
    `;
    await performQuery(deleteQuery, [{is_deleted: 1, updated_at: currentTime, updated_by: req?.user?.id}, id]);

    return sendResponse(res, {}, "Passport deleted successfully", 200);
  } catch (error) {
    return sendErrorResponse(res, error, "Error while deleting passport");
  }
};
