const { sendErrorResponse, sendResponse } = require("../../utils");
const moment = require("moment-timezone");
const { performQuery } = require("../../utils/db");
const { getSystemTime } = require("../../functions/getTimezone");
const { tables } = require("../../utils/tables");

// CREATE Emergency Contact
module.exports.createEmergencyContact = async (req, res) => {
  try {
    const {
      person_id,
      name,
      relation,
      phone,
      alternate_phone,
      address_line_1,
      address_line_2,
      address_line_3,
      locality,
      dependent_locality,
      administrative_area,
      sub_administrative_area,
      postal_code,
      country_code,
      notes,
      same_address,
    } = req.body;

    // Validation
    if (
      !person_id || !name || !relation || !phone || !same_address ||
      (same_address == 2 && (!address_line_1 || !locality || !administrative_area || !sub_administrative_area || !country_code))
    ) {
      return sendErrorResponse(
        res,
        "Required fields are missing",
        "Validation Error",
        400
      );
    }

    // Check if person exists
    const personExists = await performQuery(
      `SELECT id FROM ${tables.per_person} WHERE id = ? AND is_deleted = 2`,
      [person_id]
    );
    if (!personExists || personExists.length === 0) {
      return sendErrorResponse(res, "Person not found", "Person not found");
    }

    // Calculate completion percentage
    const totalFields = 16;
    let filledFields = 0;
    const checkFields = [
      person_id,
      name,
      relation,
      phone,
      alternate_phone,
      address_line_1,
      address_line_2,
      address_line_3,
      locality,
      dependent_locality,
      administrative_area,
      sub_administrative_area,
      postal_code,
      country_code,
      notes,
      same_address
    ];
    checkFields.forEach((f) => {
      if (f) filledFields++;
    });

    const completion = Math.round((filledFields / totalFields) * 10000) / 100;
    const is_completed = completion === 100 ? 1 : 2;

    // Get system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert record
    const insertResult = await performQuery(
      `INSERT INTO ${tables.per_emergency_contact} SET ?`,
      {
        person_id,
        name,
        relation,
        phone,
        alternate_phone,
        address_line_1,
        address_line_2,
        address_line_3,
        locality,
        dependent_locality,
        administrative_area,
        sub_administrative_area,
        postal_code,
        country_code,
        notes,
        same_address: same_address || 0,
        completion,
        is_completed,
        created_at: currentTime,
        created_by: req?.user?.id,
        updated_at: currentTime,
        updated_by: req?.user?.id,
      }
    );

    return sendResponse(
      res,
      { id: insertResult.insertId },
      "Address record created successfully",
      201
    );
  } catch (error) {
    console.log("Error in createAddressRecord:", error);
    return sendErrorResponse(
      res,
      error,
      "Failed to create address record",
      500
    );
  }
};

// UPDATE Emergency Contact
module.exports.updateEmergencyContact = async (req, res) => {
  try {
    const {
      id,
      name,
      relation,
      phone,
      alternate_phone,
      address_line_1,
      address_line_2,
      address_line_3,
      locality,
      dependent_locality,
      administrative_area,
      sub_administrative_area,
      postal_code,
      country_code,
      notes,
      same_address,
    } = req.body;

    if (
      !id || !name || !relation || !phone || !same_address ||
      (same_address == 2 && (!address_line_1 || !locality || !administrative_area || !sub_administrative_area || !country_code))
    ) {
      return sendErrorResponse(
        res,
        "Required Fields are missing",
        "Validation Error",
        400
      );
    }

    // Fetch existing record
    const existing = await performQuery(
      `SELECT * FROM ${tables.per_emergency_contact} WHERE id = ? AND is_deleted = 2`,
      [id]
    );
    if (existing.length === 0) {
      return sendErrorResponse(
        res,
        "Address record not found",
        "Not Found",
        404
      );
    }
    const old = existing[0];

    // Calculate completion percentage
    const totalFields = 16;
    let filledFields = 0;
    const checkFields = [
      old.person_id,
      name || old.name,
      relation || old.relation,
      phone || old.phone,
      alternate_phone || old.alternate_phone,
      address_line_1 || old.address_line_1,
      address_line_2 || old.address_line_2,
      address_line_3 || old.address_line_3,
      locality || old.locality,
      dependent_locality || old.dependent_locality,
      administrative_area || old.administrative_area,
      sub_administrative_area || old.sub_administrative_area,
      postal_code || old.postal_code,
      country_code || old.country_code,
      notes || old.notes,
      same_address || old.same_address
    ];
    checkFields.forEach((f) => {
      if (f) filledFields++;
    });

    const completion = Math.round((filledFields / totalFields) * 10000) / 100;
    const is_completed = completion === 100 ? 1 : 2;

    // Get system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Update record (soft delete old) to Maintain history
    await performQuery(
      `UPDATE ${tables.per_emergency_contact} SET ? WHERE id = ?`,
      [
        {
          is_deleted: 1,
          updated_at: currentTime,
          updated_by: req?.user?.id,
        },
        id,
      ]
    );

    // Insert new updated record
    await performQuery(
      `INSERT INTO ${tables.per_emergency_contact} SET ?`,
      {
        person_id: old.person_id,
        name: name || old.name,
        relation: relation || old.relation,
        phone: phone || old.phone,
        alternate_phone: alternate_phone || old.alternate_phone,
        address_line_1: address_line_1 || old.address_line_1,
        address_line_2: address_line_2 || old.address_line_2,
        address_line_3: address_line_3 || old.address_line_3,
        locality: locality || old.locality,
        dependent_locality: dependent_locality || old.dependent_locality,
        administrative_area: administrative_area || old.administrative_area,
        sub_administrative_area: sub_administrative_area || old.sub_administrative_area,
        postal_code: postal_code || old.postal_code,
        country_code: country_code || old.country_code,
        notes: notes || old.notes,
        same_address: same_address || old.same_address,
        completion: completion,
        is_completed: is_completed,
        created_at: currentTime,
        created_by: req?.user?.id,
        updated_at: currentTime,
        updated_by: req?.user?.id,
      }
    )

    return sendResponse(res, [], "Address record updated successfully", 200);
  } catch (error) {
    console.log("Error in updateAddressRecord:", error);
    return sendErrorResponse(
      res,
      error,
      "Failed to update address record",
      500
    );
  }
};

// GET Emergency Contact
module.exports.getEmergencyContact = async (req, res) => {
  try {
    const { person_id, person_id_external } = req.query;
    let query = `
    SELECT pem.* 
    FROM ${tables.per_emergency_contact} pem 
    LEFT JOIN ${tables.per_person} per ON pem.person_id = per.id
    WHERE pem.is_deleted = 2`;

    if (person_id) query += ` AND pem.person_id = ${person_id}`;
    if (person_id_external) query += ` AND per.person_id_external = '${person_id_external}'`;

    const result = await performQuery(query, []);
    return sendResponse(
      res,
      result,
      "Address record(s) retrieved successfully",
      200
    );
  } catch (error) {
    console.log("Error in getAddressRecord:", error);
    return sendErrorResponse(
      res,
      error,
      "Failed to get address record(s)",
      500
    );
  }
};

// DELETE Emergency Contact
module.exports.deleteEmergencyContact = async (req, res) => {
  try {
    const { id } = req.query;

    if (!id) {
      return sendErrorResponse(
        res,
        "Address ID is required",
        "Validation Error",
        400
      );
    }

    // Soft delete
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    await performQuery(
      `UPDATE ${tables.per_emergency_contact} SET ? WHERE id = ?`,
      [
        {
          is_deleted: 1,
          updated_at: currentTime,
          updated_by: req?.user?.id,
        },
        id,
      ]
    );

    return sendResponse(res, [], "Address record deleted successfully", 200);
  } catch (error) {
    console.log("Error in deleteAddressRecord:", error);
    return sendErrorResponse(
      res,
      error,
      "Failed to delete address record",
      500
    );
  }
};
