const { sendErrorResponse, sendResponse, } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getTimezone, getSystemTime } = require("../../functions/getTimezone.js");
const fs = require('fs');
const path = require('path');

// API to Add Educational Documents --> POST /personRecord/educational-documents
module.exports.addEducationalDocuments = async (req, res) => {
    const uploadedFiles = [];
    try {
        // Get variables from Request Body
        const {
            person_id,
            course_type,
            document_name,
            degree_title,
            start_date,
            end_date,
            status = 2,
            grade,
            institute_name,
            major_subject,
            document_no
        } = req.body;

        // Validate required fields
        if (
            !person_id || !course_type || !document_name || 
            !start_date || !institute_name || !major_subject
        ) {
            return sendErrorResponse(res, "Missing required fields", "Mandatory fields are missing", 400);
        }
        if(status == 1 && (
            !end_date || !grade || !document_no
        )) {
            return sendErrorResponse(res, "Missing required fields", "Mandatory fields are missing", 400);
        }        

        // Handle file uploads
        let baseURL = `${req.protocol}://${req.get("host")}/uploads`;
        const attachment_1 = req.files?.attachment_1 ? req.files.attachment_1[0].filename : null;
        const attachment_2 = req.files?.attachment_2 ? req.files.attachment_2[0].filename : null;
        if (attachment_1) uploadedFiles.push(attachment_1);
        if (attachment_2) uploadedFiles.push(attachment_2);
        if (status == 1 && (!attachment_1 && !attachment_2)) {
            return sendErrorResponse(res, "At least one attachment is required", "At Least one attachment is Required", 400);
        }

        // Check is person exists
        const personExists = await performQuery(`SELECT id FROM ${tables.per_person} WHERE id = ?`, [person_id]);
        if (personExists.length === 0) {
            return sendErrorResponse(res, "Person not found", "Person not Found", 404);
        }

        // check is course_type valid
        const courseTypeExists = await performQuery(
            `SELECT * FROM ${tables.picklist_master} WHERE id = ? AND picklist_name = 'course_type' AND is_deleted = 2`,
            [course_type]
        );
        if (courseTypeExists.length === 0) {
            return sendErrorResponse(res, "Invalid course type", "Invalid course type", 400);
        }

        if (degree_title) {
            // check is degree_title valid
            const degreeTitleExists = await performQuery(
                `SELECT * FROM ${tables.picklist_master} WHERE id = ? AND picklist_name = 'degree_title' AND is_deleted = 2`,
                [degree_title]
            );
            if (degreeTitleExists.length === 0) {
                return sendErrorResponse(res, "Invalid degree title", "Invalid degree title", 400);
            }
        }

        // Apply validation for in-progress education records
        const inProgressEducationCount = await performQuery(
            `SELECT COUNT(*) AS count FROM ${tables.per_educational_detail} WHERE person_id = ${person_id} AND status = 2 AND is_deleted = 2`
        );
        if (inProgressEducationCount[0].count >= 1 && status == 2) {
            return sendErrorResponse(res, "Maximum in-progress education records reached", "Maximum in-progress education records reached", 400);
        }

        // Apply validation for completed education records check is document no already exists
        if (status == 1) {
            const documentNoExists = await performQuery(
                `SELECT id FROM ${tables.per_educational_detail} WHERE document_no = ? AND is_deleted = 2`,
                [document_no]
            );
            if (documentNoExists.length > 0) {
                return sendErrorResponse(res, "Document number already exists", "Document Already Exists", 400);
            }
        }

        // Calculate completion %
        const total_fields = 11; 
        let filled_fields = 0;
        const checkFilled = (v) => { if (v) filled_fields++; };
        [
            person_id,
            course_type,
            document_name,
            degree_title,
            start_date,
            end_date,
            grade,
            institute_name,
            major_subject,
            document_no,
            attachment_1
        ].forEach(checkFilled);

        const completion = Math.round((filled_fields / total_fields) * 100);
        const is_completed = completion === 100 ? 2 : 1; // 2 = completed, 1 = not completed


        // Get system time
        const systemTime = await getSystemTime();
        const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

        // Create File URL
        const attachment1URL = attachment_1 ? `${baseURL}/${attachment_1}` : null;
        const attachment2URL = attachment_2 ? `${baseURL}/${attachment_2}` : null;

        // Insert Educational Details Record
        const insertEducationResult = await performQuery(
            `INSERT INTO ${tables.per_educational_detail} SET ?`,
            {
                person_id: person_id,
                course_type: course_type,
                document_name: document_name || null,
                degree_title: degree_title || null,
                start_date: start_date,
                end_date: end_date || null,
                status: status,
                grade: grade || null,
                institute_name: institute_name,
                major_subject: major_subject,
                document_no: document_no || null,
                attachment_1: attachment1URL,
                attachment_2: attachment2URL,
                completion: completion,
                is_completed: is_completed,
                created_at: currentTime,
                created_by: req?.user?.id,
                updated_at: currentTime,
                updated_by: req?.user?.id,
            }
        );

        // Send Success Response
        return sendResponse(res, [], "Educational documents added successfully", 201);

    } catch (error) {
        console.log("Error in Adding Educational Documents: ", error);
        return sendErrorResponse(res, "Failed to add educational documents", "Failed to add educational documents", 500);
    }
};

// API to Update Educational Documents --> PUT /personRecord/educational-documents
module.exports.updateEducationalDocuments = async (req, res) => {
    const uploadedFiles = [];
    try {
        const { 
            id,
            course_type,
            document_name,
            degree_title,
            start_date,
            end_date,
            status = 2,
            grade,
            institute_name,
            major_subject,
            document_no
        } = req.body;

        // Validate required fields
        if (!id || !course_type || !start_date || !status || !institute_name || !major_subject) {
            return sendErrorResponse(res, "Required fields are missing", "Validation Error", 400);
        }
        if(status == 1 && (
            !end_date || !grade || !document_no
        )) {
            return sendErrorResponse(res, "Missing required fields", "Validation Error", 400);
        }

        // Handle file uploads
        console.log("Uploaded Files: ", req.files?.attachment_1);
        
        let baseURL = `${req.protocol}://${req.get("host")}/uploads`;
        const attachment_1 = req.files?.attachment_1 ? req.files.attachment_1[0].filename : null;
        const attachment_2 = req.files?.attachment_2 ? req.files.attachment_2[0].filename : null;
        if (attachment_1) uploadedFiles.push(attachment_1);
        if (attachment_2) uploadedFiles.push(attachment_2);

        console.log("attachment_1: ", attachment_1);
        
        // Check if the educational record exists
        const educationRecord = await performQuery(
            `SELECT * FROM ${tables.per_educational_detail} WHERE id = ? AND is_deleted = 2`,
            [id]
        );
        if (educationRecord.length === 0) {
            return sendErrorResponse(res, "Educational record not found", "Educational record not found", 404);
        }
        const existingRecord = educationRecord[0];

        // check is course_type valid
        const courseTypeExists = await performQuery(
            `SELECT * FROM ${tables.picklist_master} WHERE id = ? AND picklist_name = 'course_type' AND is_deleted = 2`,
            [course_type]
        );
        if (courseTypeExists.length === 0) {
            return sendErrorResponse(res, "Invalid course type", "Invalid Cource Type", 400);
        }

        if (degree_title) {
            // check is degree_title valid
            const degreeTitleExists = await performQuery(
                `SELECT * FROM ${tables.picklist_master} WHERE id = ? AND picklist_name = 'degree_title' AND is_deleted = 2`,
                [degree_title]
            );
            if (degreeTitleExists.length === 0) {
                return sendErrorResponse(res, "Invalid degree title", "Invalid Degree Title", 400);
            }
        }

        // Apply validation for in-progress education records
        const inProgressEducationCount = await performQuery(
            `SELECT COUNT(*) AS count FROM ${tables.per_educational_detail} WHERE person_id = ${existingRecord.person_id} AND status = 2 AND id<> ${id} AND is_deleted = 2`
        );
        if (inProgressEducationCount[0].count >= 1  && status == 2) {
            return sendErrorResponse(res, "Maximum in-progress education records reached", "Maximum In-Progress Education Records Reached", 400);
        }

        // Apply validation for completed education records check is document no already exists
        if (status == 1) {
            const documentNoExists = await performQuery(
                `SELECT id FROM ${tables.per_educational_detail} WHERE document_no = ? AND id <> ? AND is_deleted = 2`,
                [document_no, id]
            );
            if (documentNoExists.length > 0) {
                return sendErrorResponse(res, "Document number already exists", "Document Number Already Exists", 400);
            }
        }
                
        // Calculate completion %
        const total_fields = 11; 
        let filled_fields = 0;
        const checkFilled = (v) => { if (v) filled_fields++; };
        [
            existingRecord.person_id,
            course_type || existingRecord.course_type,
            document_name || existingRecord.document_name,
            degree_title || existingRecord.degree_title,
            start_date || existingRecord.start_date,
            end_date || existingRecord.end_date,
            grade || existingRecord.grade,
            institute_name || existingRecord.institute_name,
            major_subject || existingRecord.major_subject,
            document_no || existingRecord.document_no,
            attachment_1 || existingRecord.attachment_1,
        ].forEach(checkFilled);

        const completion = Math.round((filled_fields / total_fields) * 100);
        const is_completed = completion === 100 ? 2 : 1; // 2 = completed, 1 = not completed


        // Get system time
        const systemTime = await getSystemTime();
        const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

        // Create File URLs
        const attachment1URL = attachment_1 ? `${baseURL}/${attachment_1}` : null;
        const attachment2URL = attachment_2 ? `${baseURL}/${attachment_2}` : null;
        console.log("attachment1URL: ", attachment1URL);

        // Mark previous record as deleted to maintain history
        const updateEducationRecord = await performQuery(
            `UPDATE ${tables.per_educational_detail} SET ? WHERE id = ?`,
            [
                {
                    is_deleted: 1,
                    updated_at: currentTime,
                    updated_by: req?.user?.id,
                },
                id
            ]
        );

        // Insert Educational Details Record
        const insertEducationResult = await performQuery(
            `INSERT INTO ${tables.per_educational_detail} SET ?`,
            {
                person_id: existingRecord.person_id,
                course_type: course_type || existingRecord.course_type,
                document_name: document_name || existingRecord.document_name,
                degree_title: degree_title || existingRecord.degree_title,
                start_date: start_date || existingRecord.start_date,
                end_date: end_date || existingRecord.end_date,
                status: status || existingRecord.status,
                grade: grade || existingRecord.grade,
                institute_name: institute_name || existingRecord.institute_name,
                major_subject: major_subject || existingRecord.major_subject,
                document_no: document_no || existingRecord.document_no,
                attachment_1: attachment1URL || existingRecord.attachment_1,
                attachment_2: attachment2URL || existingRecord.attachment_2,
                completion: completion,
                is_completed: is_completed,
                created_at: currentTime,
                created_by: req?.user?.id,
                updated_at: currentTime,
                updated_by: req?.user?.id,
            }
        );

        // Send Success Response
        return sendResponse(res, [], "Educational documents updated successfully", 200);

    } catch (error) {
        console.log("Error in Updating Educational Documents: ", error);
        return sendErrorResponse(res, "Failed to update educational documents", "Failed to update educational documents", 500);
    }
}


// API to Select Educational Documents --> GET /personRecord/educational-documents
module.exports.getEducationalDocuments = async (req, res) => {
    try {
        // Get query parameters
        const { person_id, person_id_external } = req.query;

        // Build the query
        let query = `
            SELECT ped.*, pp.display_name AS person_name, pp.person_id_external, ct.picklist_option AS course_type_value, dt.picklist_option AS degree_title_value
            FROM ${tables.per_educational_detail} AS ped
            LEFT JOIN ${tables.per_person} AS pp ON ped.person_id = pp.id
            Left Join ${tables.picklist_master} ct ON ct.id = ped.course_type
            Left Join ${tables.picklist_master} dt ON dt.id = ped.degree_title	
            WHERE ped.is_deleted = 2`;

        if (person_id) {
            query += ` AND ped.person_id = ${person_id}`;
        }
        if (person_id_external) {
            query += ` AND pp.person_id_external = '${person_id_external}'`;
        }

        // Sort by most recent id
        query += ` ORDER BY ped.id DESC`;

        // Execute the query
        const result = await performQuery(query);

        return sendResponse(res, result, "Educational documents retrieved successfully", 200);
    } catch (error) {
        console.log("Error in Getting Educational Documents: ", error);
        return sendErrorResponse(res, error, "Failed to get educational documents", 500);
    }
}

// API to Delete Educational Documents --> DELETE /personRecord/educational-documents
module.exports.deleteEducationalDocuments = async (req, res) => {
    try {
        const { id } = req.query;

        // Validate request
        if (!id) {
            return sendErrorResponse(res, "Invalid request", "Educational document ID is required", 400);
        }

        // Build the query
        const query = `UPDATE ${tables.per_educational_detail} SET is_deleted = 1 WHERE id = ${id}`;

        // Execute the query
        const result = await performQuery(query);

        return sendResponse(res, result, "Educational document deleted successfully", 200);
    } catch (error) {
        console.log("Error in Deleting Educational Document: ", error);
        return sendErrorResponse(res, error, "Failed to delete educational document", 500);
    }
}
