const {
  sendErrorResponse,
  sendResponse,
} = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");


// CREATE Department
module.exports.createDepartment = async (req, res) => {
  try {
    const {
      department_id_external,
      division_id,
      effective_start_date,
      description,
      description_arabic,
      cost_center_id
    } = req.body;

    if (!department_id_external || !division_id || !effective_start_date) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check if division exists
    const isDivision = await performQuery(
      `SELECT * FROM ${tables.org_division} WHERE id = ? AND is_deleted = 2`,
      [division_id]
    );
    if (isDivision.length === 0) {
      const errorMessage = "Division ID not found in Division table";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check is Cost Center exists
    const isCostCenter = await performQuery(
      `SELECT * FROM ${tables.org_cost_center} WHERE id = ? AND is_deleted = 2`,
      [cost_center_id]
    );
    if (isCostCenter.length === 0) {
      const errorMessage = "Cost Center ID not found in Cost Center table";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check for duplicate department_id_external
    const existingDept = await performQuery(
      `SELECT id FROM ${tables.org_department} WHERE department_id_external = ? AND is_deleted = 2`,
      [department_id_external]
    );
    if (existingDept && existingDept.length > 0) {
      const errorMessage = "Department ID already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Get system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert Record
    await performQuery(`INSERT INTO ${tables.org_department} SET ?`, {
      department_id_external,
      division_id,
      effective_start_date,
      description,
      description_arabic,
      cost_center_id,
      created_at: currentTime,
      created_by: req?.user?.id,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    });

    return sendResponse(res, {}, "Department created successfully", 201);
  } catch (error) {
    console.error("Error while creating Department:", error);
    return sendErrorResponse(res, error, "Error while creating Department");
  }
};


// READ Department(s)
module.exports.getDepartment = async (req, res) => {
  try {
    const { id, department_id_external, division_id, division_id_external, business_unit_id, bu_id_external, entity_id, entity_id_external } = req.query;

    let query = `
      SELECT 
        dep.*,
        division.division_id_external, division.description AS division_description,
        bu.bu_id_external, bu.description AS business_unit_description,
        le.entity_id_external, le.name AS legal_entity_name,
        cc.cost_center_id_external, cc.name_us AS cost_center_name, cc.description_us AS cost_center_description, cc.gl_statement_code
      FROM ${tables.org_department} dep
      LEFT JOIN ${tables.org_division} division ON dep.division_id = division.id AND division.is_deleted = 2
      LEFT JOIN ${tables.org_business_unit} bu ON division.business_unit_id = bu.id AND bu.is_deleted = 2
      LEFT JOIN ${tables.org_legal_entity} le ON bu.entity_id = le.id AND le.is_deleted = 2
      LEFT JOIN ${tables.org_cost_center} cc ON dep.cost_center_id = cc.id AND cc.is_deleted = 2
      WHERE dep.is_deleted = 2
    `;

    if (id) query += ` AND dep.id = ${id}`;
    if (department_id_external) query += ` AND dep.department_id_external = '${department_id_external}'`;
    if (division_id) query += ` AND dep.division_id = ${division_id}`;
    if (division_id_external) query += ` AND division.division_id_external = '${division_id_external}'`;
    if (business_unit_id) query += ` AND bu.id = ${business_unit_id}`;
    if (bu_id_external) query += ` AND bu.bu_id_external = '${bu_id_external}'`;
    if (entity_id) query += ` AND le.id = ${entity_id}`;
    if (entity_id_external) query += ` AND le.entity_id_external = '${entity_id_external}'`;

    query += ` ORDER BY dep.id DESC`;

    const departments = await performQuery(query);
    return sendResponse(res, departments, "Department(s) fetched successfully", 200);
  } catch (error) {
    console.error("Error while fetching Department(s):", error);
    return sendErrorResponse(res, error, "Error while fetching Department(s)");
  }
};


// UPDATE Department
module.exports.updateDepartment = async (req, res) => {
  try {
    const {
      id,
      department_id_external,
      division_id,
      effective_start_date,
      description,
      description_arabic,
      cost_center_id,
      status,
    } = req.body;

    if (!id || !department_id_external || !division_id || !effective_start_date) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check if record exists
    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.org_department} WHERE id = ? AND is_deleted = 2`,
      [id]
    );
    if (!existingRecord || existingRecord.length === 0) {
      const errorMessage = "Department record not found";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    const oldRecord = existingRecord[0];

    // Check for duplicate department_id_external
    const duplicateCheck = await performQuery(
      `SELECT id FROM ${tables.org_department} WHERE department_id_external = ? AND id <> ? AND is_deleted = 2`,
      [department_id_external, id]
    );
    if (duplicateCheck && duplicateCheck.length > 0) {
      const errorMessage = "Department ID already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check is Cost Center exists
    const isCostCenter = await performQuery(
      `SELECT * FROM ${tables.org_cost_center} WHERE id = ? AND is_deleted = 2`,
      [cost_center_id]
    );
    if (isCostCenter.length === 0) {
      const errorMessage = "Cost Center ID not found in Cost Center table";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Get system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Update Record
    await performQuery(`UPDATE ${tables.org_department} SET ? WHERE id = ?`, [
      {
        effective_start_date: effective_start_date || oldRecord.effective_start_date,
        description: description || oldRecord.description,
        description_arabic: description_arabic || oldRecord.description_arabic,
        cost_center_id: cost_center_id || oldRecord.cost_center_id,
        status: status || oldRecord.status,
        updated_at: currentTime,
        updated_by: req?.user?.id,
      },
      id,
    ]);

    return sendResponse(res, {}, "Department updated successfully", 200);
  } catch (error) {
    console.error("Error while updating Department:", error);
    return sendErrorResponse(res, error, "Error while updating Department");
  }
};


// DELETE Department (Soft Delete)
// module.exports.deleteDepartment = async (req, res) => {
//   try {
//     const { id } = req.query;

//     if (!id) {
//       const errorMessage = "Department ID is required";
//       return sendErrorResponse(res, errorMessage, errorMessage);
//     }

//     // Get system time
//     const systemTime = await getSystemTime();
//     const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

//     // Soft delete
//     await performQuery(
//       `UPDATE ${tables.org_department} SET ? WHERE id = ?`,
//       [
//         {
//           is_deleted: 1,
//           updated_at: currentTime,
//           updated_by: req?.user?.id,
//         },
//         id,
//       ]
//     );

//     return sendResponse(res, {}, "Department deleted successfully", 200);
//   } catch (error) {
//     console.error("Error while deleting Department:", error);
//     return sendErrorResponse(res, error, "Error while deleting Department");
//   }
// };
