const { sendErrorResponse, sendResponse, } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");

// --> Create Holiday Assignments
module.exports.createWorkSchedule = async (req, res) => {
    try {
        // Extract required fields from request body
        const {
            schedule_id_external,
            external_name,
            starting_date,
            country,
            model,
            cross_midnight_allowed,
            user,
            average_hours_per_day,
            average_hours_per_week,
            average_hours_per_month,
            average_hours_per_year,
            average_working_days_per_week,
            shift_classification,
            time_recording_variant,
        } = req.body;

        // Validate required fields
        if (
            !schedule_id_external || !external_name || !starting_date || !country || !model ||
            cross_midnight_allowed === undefined || time_recording_variant === undefined
        ) {
            return sendErrorResponse(
                res,
                "Schedule ID, External Name, Starting Date, Country, and Model are required",
                "Schedule ID, External Name, Starting Date, Country, and Model are required"
            );
        }

        // Check for existing active work schedule with the same external ID
        const existingSchedule = await performQuery(
            `SELECT * FROM ${tables.work_schedule} WHERE schedule_id_external = ? AND is_active = 1`,
            [schedule_id_external]
        );
        if (existingSchedule.length > 0) {
            return sendErrorResponse(res, "Work Schedule with this External ID already exists", "Work Schedule with this External ID already exists");
        }

        // Check if same external name exists for the country
        const existingNameForCountry = await performQuery(
            `SELECT * FROM ${tables.work_schedule} WHERE external_name = ? AND country = ? AND is_active = 1`,
            [external_name, country]
        );
        if (existingNameForCountry.length > 0) {
            return sendErrorResponse(res, "Work Schedule with this External Name already exists for the specified country", "Work Schedule with this External Name already exists for the specified country");
        }

        // Check is model exist in picklist
        const checkModel = await performQuery(
            `SELECT * FROM ${tables.picklist_master} WHERE (picklist_id = 18 and id = ?) AND is_deleted = 2`,
            [model]
        );
        if (checkModel.length === 0) {
            return sendErrorResponse(res, "Invalid model value", "Invalid model value");
        }

        // Check is time_recording_variant exist in picklist
        const checkTimeRecordingVarient = await performQuery(
            `SELECT * FROM ${tables.picklist_master} WHERE (picklist_id = 19 and id = ?) AND is_deleted = 2`,
            [time_recording_variant]
        );
        if (checkTimeRecordingVarient.length === 0) {
            return sendErrorResponse(res, "Invalid time recording varient value", "Invalid time recording varient value");
        }

        // Get current system time
        const systemTime = await getSystemTime();
        const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

        // Insert new work schedule record
        const result = await performQuery(
            `INSERT INTO ${tables.work_schedule} SET ?`, {
                sequence_no: 1,
                schedule_id_external,
                external_name,
                starting_date,
                country,
                model,
                cross_midnight_allowed,
                user,
                average_hours_per_day: average_hours_per_day || null,
                average_hours_per_week: average_hours_per_week || null,
                average_hours_per_month: average_hours_per_month || null,
                average_hours_per_year: average_hours_per_year || null,
                average_working_days_per_week: average_working_days_per_week || null,
                shift_classification: shift_classification || null,
                time_recording_variant,
                created_by: req?.user?.id,
                created_at: currentTime,
                updated_by: req?.user?.id,
                updated_at: currentTime,
            }
        );

        // Send success response
        return sendResponse(res, {}, "Work Schedule Created Successfully", 201);

    } catch (error) {
        console.error("Error While Creating Work Schedule: ", error);
        return sendErrorResponse(res, "Internal Server Error", "Internal Server Error");
    }
}

module.exports.updateWorkSchedule = async (req, res) => {
    try {
        // Extract required fields from request body
        const {
            id,
            external_name,
            starting_date,
            country,
            model,
            cross_midnight_allowed,
            user,
            average_hours_per_day,
            average_hours_per_week,
            average_hours_per_month,
            average_hours_per_year,
            average_working_days_per_week,
            shift_classification,
            time_recording_variant,
        } = req.body;

        // Validate required fields
        if (
            !id || !external_name || !starting_date || !country || !model ||
            cross_midnight_allowed === undefined || time_recording_variant === undefined
        ) {
            return sendErrorResponse(
                res,
                "Schedule ID, External Name, Starting Date, Country, and Model are required",
                "Schedule ID, External Name, Starting Date, Country, and Model are required"
            );
        }

        // Check is Record exist against the id
        const existingRecord = await performQuery(
            `SELECT * FROM ${tables.work_schedule} WHERE id = ? AND is_active = 1`,
            [id]
        );
        if (existingRecord.length === 0) {
            return sendErrorResponse(res, "No active Work Schedule found with the provided ID", "No active Work Schedule found with the provided ID");
        }

        // Check if same external name exists for the country excluding current record
        const existingNameForCountry = await performQuery(
            `SELECT * FROM ${tables.work_schedule} WHERE external_name = ? AND country = ? AND id <> ? AND is_active = 1`,
            [external_name, country, id]
        );
        if (existingNameForCountry.length > 0) {
            return sendErrorResponse(res, "Work Schedule with this External Name already exists for the specified country", "Work Schedule with this External Name already exists for the specified country");
        }
        
        // Check is model exist in picklist
        const checkModel = await performQuery(
            `SELECT * FROM ${tables.picklist_master} WHERE (picklist_id = 18 and id = ?) AND is_deleted = 2`,
            [model]
        );
        if (checkModel.length === 0) {
            return sendErrorResponse(res, "Invalid model value", "Invalid model value");
        }

        // Check is time_recording_variant exist in picklist
        const checkTimeRecordingVarient = await performQuery(
            `SELECT * FROM ${tables.picklist_master} WHERE (picklist_id = 19 and id = ?) AND is_deleted = 2`,
            [time_recording_variant]
        );
        if (checkTimeRecordingVarient.length === 0) {
            return sendErrorResponse(res, "Invalid time recording varient value", "Invalid time recording varient value");
        }

        // Get current system time
        const systemTime = await getSystemTime();
        const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

        // Update existing work schedule record is_active from 1 to 2 to make it inactive
        await performQuery(
            `UPDATE ${tables.work_schedule} SET is_active = 2, updated_by = ?, updated_at = ? WHERE id = ?`,
            [req?.user?.id, currentTime, id]
        );

        // Insert new work schedule record with updated data
        const result = await performQuery(
            `INSERT INTO ${tables.work_schedule} SET ?`, {
                sequence_no: Number(existingRecord[0].sequence_no) + 1,
                schedule_id_external: existingRecord[0].schedule_id_external,
                external_name,
                starting_date,
                country,
                model,
                cross_midnight_allowed,
                user,
                average_hours_per_day: average_hours_per_day || null,
                average_hours_per_week: average_hours_per_week || null,
                average_hours_per_month: average_hours_per_month || null,
                average_hours_per_year: average_hours_per_year || null,
                average_working_days_per_week: average_working_days_per_week || null,
                shift_classification: shift_classification || null,
                time_recording_variant,
                created_by: req?.user?.id,
                created_at: currentTime,
                updated_by: req?.user?.id,
                updated_at: currentTime,
            }
        );

        // Send success response
        return sendResponse(res, {}, "Work Schedule Updated Successfully", 200);

    } catch (error) {
        console.log("Error While Updating Work Schedule: ", error);
        return sendErrorResponse(res, "Internal Server Error", "Internal Server Error");
    }
}

module.exports.getWorkSchedules = async (req, res) => {
    try {
        // Extract query parameters
        const { id, schedule_id_external } = req.query;

        // Build query dynamically based on provided parameters
        let query = `
            SELECT 
                ws.id, ws.schedule_id_external, 
                ws.external_name, ws.starting_date, 
                ws.country, 
                ws.model, pm_model.picklist_option AS model_value, 
                ws.cross_midnight_allowed, 
                ws.user, 
                ws.average_hours_per_day, ws.average_hours_per_week, 
                ws.average_hours_per_month, ws.average_hours_per_year, 
                ws.average_working_days_per_week, ws.shift_classification, 
                ws.time_recording_variant, pm_time_recording.picklist_option AS time_recording_variant_value,
                ws.sequence_no,
                ws.created_by, ws.created_at, 
                ws.updated_by, ws.updated_at
            FROM ${tables.work_schedule} ws 
            LEFT JOIN ${tables.picklist_master} pm_model ON ws.model = pm_model.id AND pm_model.picklist_id = 18
            LEFT JOIN ${tables.picklist_master} pm_time_recording ON ws.time_recording_variant = pm_time_recording.id AND pm_time_recording.picklist_id = 19
            WHERE ws.is_active = 1`;

        if (id) {
            query += ` AND ws.id = ${id}`;
        }
        if (schedule_id_external) {
            query += ` AND ws.schedule_id_external = '${schedule_id_external}'`;
        }

        // Execute query to fetch work schedules
        const workSchedules = await performQuery(query);

        // Send success response with fetched work schedules
        return sendResponse(res, workSchedules, "Work Schedules Fetched Successfully", 200);

    } catch (error) {
        console.log("Error While Fetching Work Schedules: ", error);
        return sendErrorResponse(res, "Internal Server Error", "Internal Server Error");       
    }
}