const { sendErrorResponse, sendResponse, } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");

// --> Create Holidays
module.exports.createHolidays = async (req, res) => {
  try {
    const {
      holiday_code,
      name,
      country
    } = req.body;

    if (!name || !holiday_code || !country) {
      return sendErrorResponse(
        res,
        "Holiday name, code, and country are required",
        "Holiday name, code, and country are required"
      );
    }

    // Check for existing active holiday with the same code
    const existingHoliday = await performQuery(
      `SELECT * FROM ${tables.holidays} WHERE holiday_code = ? AND is_active = 1`,
      [holiday_code]
    );
    if (existingHoliday.length > 0) {
      return sendErrorResponse(res, "Holiday code already exists", "Holiday code already exists");
    }

    // Check for Existing Holiday with the same name & country
    const existingHolidayByNameCountry = await performQuery(
      `SELECT * FROM ${tables.holidays} WHERE name = ? AND country = ? AND is_active = 1`,
      [name, country]
    );
    if (existingHolidayByNameCountry.length > 0) {
      return sendErrorResponse(res, "Holiday with the same name already exists in the specified country", "Holiday with the same name already exists in the specified country");
    }

    // Get current system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    await performQuery(
      `INSERT INTO ${tables.holidays} SET ?`,
      {
        holiday_code,
        name,
        country: country || null,
        created_by: req?.user?.id,
        created_at: currentTime,
        updated_by: req?.user?.id,
        updated_at: currentTime
      }
    );

    return sendResponse(res, {}, "Holiday created successfully", 200);

  } catch (error) {
    console.error("Error While Creating Holiday:", error);
    return sendErrorResponse(res, error, "Error while creating holiday");
  }
};

// --> Get Holidays 
module.exports.getHolidays = async (req, res) => {
  try {
    const { id } = req.query;

    let query = `
      SELECT *
      FROM ${tables.holidays}
      WHERE is_active = 1
    `;

    if (id) query += ` AND id = ${id}`;
    query += ` ORDER BY id DESC`;

    const result = await performQuery(query);
    return sendResponse(res, result, "Holidays fetched successfully", 200);

  } catch (error) {
    console.error("Error While Fetching Holidays:", error);
    return sendErrorResponse(res, error, "Error while fetching holidays");
  }
};

// --> Update Holidays 
module.exports.updateHolidays = async (req, res) => {
  try {
    const {
      id,
      name,
      country,
    } = req.body;

    if (!id || !name || !country) {
      return sendErrorResponse(
        res,
        "ID, Name, Holiday Code, and Country are required",
        "ID, Name, Holiday Code, and Country are required"
      );
    }

    // Check existing record
    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.holidays} WHERE id = ? AND is_active = 1`,
      [id]
    );
    if (existingRecord.length === 0) {
      return sendErrorResponse(res, "Holiday not found", "Holiday not found");
    }

    // Check for Existing Holiday with the same name & country (excluding current record)
    const existingHolidayByNameCountry = await performQuery(
      `SELECT * FROM ${tables.holidays} WHERE name = ? AND country = ? AND is_active = 1 AND id <> ?`,
      [name, country, id]
    );
    if (existingHolidayByNameCountry.length > 0) {
      return sendErrorResponse(res, "Holiday with the same name already exists in the specified country", "Holiday with the same name already exists in the specified country");
    }

    // Get current system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Deactivate existing record to maintain history
    await performQuery(
      `UPDATE ${tables.holidays}
       SET ?
       WHERE id = ?`,[{
        is_active: 2,
        updated_by: req?.user?.id,
        updated_at: currentTime
      }, id]
    );

    // Insert new record with updated details
    const result = await performQuery(
      `Insert INTO ${tables.holidays} SET ?`,{
        sequence_no: Number(existingRecord[0].sequence_no) + 1,
        holiday_code: existingRecord[0].holiday_code,
        name,
        country,
        created_by: req?.user?.id,
        created_at: currentTime,
        updated_by: req?.user?.id,
        updated_at: currentTime,
      }
    );

    return sendResponse(res, {}, "Holiday updated successfully", 200);

  } catch (error) {
    console.log("Error While Updating Holiday:", error);
    return sendErrorResponse(res, error, "Error while updating holiday");
  }
};
