const { sendErrorResponse, sendResponse, } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");

// --> Create Holiday calendar 
module.exports.createHolidaycalendar = async (req, res) => {
  try {
    const {
      calendar_id_external,
      name_en,
      country
    } = req.body;
    
    if (!calendar_id_external || !name_en || !country) {
        const errorMessage = "Missing required fields! "
        return sendErrorResponse(res,errorMessage , errorMessage);
    }

    // Check for existing active record
    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.holiday_calendar} WHERE calendar_id_external = ? AND is_active = 1`,
      [calendar_id_external]
    );
    if (existingRecord.length > 0) {
      return sendErrorResponse(res, "Holiday calendar with the same external ID already exists", "Holiday calendar with the same external ID already exists");
    }

    // Check for Existing Holiday with the same name & country
    const existingHolidayByNameCountry = await performQuery(
      `SELECT * FROM ${tables.holiday_calendar} WHERE name_en = ? AND country = ? AND is_active = 1`,
      [name_en, country]
    );
    if (existingHolidayByNameCountry.length > 0) {
      return sendErrorResponse(res, "Holiday calendar with the same name already exists in the specified country", "Holiday calendar with the same name already exists in the specified country");
    }

    // Get current system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    await performQuery(
      `INSERT INTO ${tables.holiday_calendar} SET ?`,
      {
        sequence_no: 1,
        calendar_id_external,
        name_en,
        country,
        is_active: 1,
        created_by: req?.user?.id,
        created_at: currentTime,
        updated_by: req?.user?.id,
        updated_at: currentTime,
      }
    );

    return sendResponse(res, {}, "Holiday calendar created successfully", 200);

  } catch (error) {
    console.error("Error While Creating Holiday calendar:", error);
    return sendErrorResponse(res, error, "Error while creating holiday calendar");
  }
}; 

// --> Get Holiday calendar
module.exports.getHolidaycalendar = async (req, res) => {
    try {
    const { id, calendar_id_external } = req.query;

    let query = `
      SELECT 
        hc.id, hc.calendar_id_external, hc.name_en, hc.country, 
        h.id AS holiday_id, h.name AS holiday_name, h.holiday_code,
        ha.id AS holiday_assignment_id, ha.date_of_holiday, 
        ha.holiday_class, pcm.picklist_option AS holiday_class_value,
        ha.holiday_category, pcm2.picklist_option AS holiday_category_value
      FROM ${tables.holiday_calendar} hc
      LEFT JOIN ${tables.holiday_assignments} ha ON ha.calendar_id_external = hc.calendar_id_external AND ha.is_active = 1
      LEFT JOIN ${tables.holidays} h ON h.holiday_code = ha.holiday_code AND ha.is_active = 1
      LEFT JOIN ${tables.picklist_master} pcm ON ha.holiday_class = pcm.id AND pcm.is_deleted = 2
      LEFT JOIN ${tables.picklist_master} pcm2 ON ha.holiday_category = pcm2.id AND pcm2.is_deleted = 2
      WHERE hc.is_active = 1
    `;

    if (id) query += ` AND hc.id = ${id}`;
    if (calendar_id_external) query += ` AND hc.calendar_id_external = '${calendar_id_external}'`;
    query += ` ORDER BY hc.id DESC`;
    const result = await performQuery(query);
    const transformedResult = await TransformData(result);
    return sendResponse(res, transformedResult, "Holiday calendar fetched successfully", 200);

  } catch (error) {
    console.error("Error While Fetching Holiday calendar", error);
    return sendErrorResponse(res, error, "Error While Fetching Holiday calendar");
  }
};

// --> Update Holiday calendar 
module.exports.updateHolidaycalendar = async (req, res) => {
  try {
    const {
      id,
      name_en,
      country,
    } = req.body;

    if (!id || !name_en || !country) {
        const errorMessage = "Missing required fields! "
        return sendErrorResponse(res,errorMessage , errorMessage);
    }

    // Check existing record
    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.holiday_calendar} WHERE id = ? AND is_active = 1`,
      [id]
    );
    if (existingRecord.length === 0) {
      return sendErrorResponse(res, "Holiday calendar not found", "Holiday calendar not found");
    }

    // Check for Existing Holiday with the same name & country (excluding current record)
    const existingHolidayByNameCountry = await performQuery(
      `SELECT * FROM ${tables.holiday_calendar} WHERE name_en = ? AND country = ? AND is_active = 1 AND id <> ?`,
      [name_en, country, id]
    );
    if (existingHolidayByNameCountry.length > 0) {
      return sendErrorResponse(res, "Holiday calendar with the same name already exists in the specified country", "Holiday calendar with the same name already exists in the specified country");
    }

    // Get current system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Deactivate existing record to maintain history
    await performQuery(
      `UPDATE ${tables.holiday_calendar}
        SET ?
        WHERE id = ?`,
      [{
        is_active: 2,
        updated_by: req?.user?.id,
        updated_at: currentTime
      }, id]
    );

    // Insert new record with updated details
    const result = await performQuery(
      `Insert INTO ${tables.holiday_calendar} SET ?`,
      {
        sequence_no: Number(existingRecord[0].sequence_no) +1,
        calendar_id_external: existingRecord[0].calendar_id_external,
        name_en,
        country,
        is_active: 1,
        created_by: req?.user?.id,
        created_at: currentTime,
        updated_by: req?.user?.id,
        updated_at: currentTime,
      }
    );

    return sendResponse(res, {}, "Holiday calendar updated successfully", 200);

  } catch (error) {
    console.log("Error While Updating Holiday calendar:", error);
    return sendErrorResponse(res, error, "Error while updating holiday calendar");
  }
};

const TransformData = async (data) => {
  const map = new Map();
  data?.forEach((row) => {
    // Grouping key: adjust if you want a different grouping logic
    const key = `${row.id}`;
    if (!map.has(key)) {
      map.set(key, {
        id: row.id,
        calendar_id_external: row.calendar_id_external,
        name_en: row.name_en,
        country: row.country,
        created_at: row.created_at,
        updated_at: row.updated_at,
        created_by: row.created_by,
        updated_by: row.updated_by,
        holiday_assignments: [],
      });
    }
    const agg = map.get(key);
    if (row.holiday_assignment_id !== null) {
      const existingHolidayAssignmentId = agg.holiday_assignments.some(item => item.id === row.holiday_assignment_id);
      if (!existingHolidayAssignmentId) {
        agg.holiday_assignments.push({
          id: row.holiday_assignment_id,
          date_of_holiday: row.date_of_holiday,
          holiday_class: row.holiday_class,
          holiday_class_value: row.holiday_class_value,
          holiday_code: row.holiday_code,
          holiday_name: row.holiday_name,
          holiday_category: row.holiday_category,
          holiday_category_value: row.holiday_category_value,
        });
      }    
    }
  });
  return Array.from(map.values());
}