const { sendErrorResponse, sendResponse, } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");

// --> Create Holiday Assignments
module.exports.createHolidayAssignments = async (req, res) => {
    try {
        const {
          calendar_id_external,
          date_of_holiday,		
          holiday_class,	
          holiday_code,		
          holiday_category 	
        } = req.body;
    
        // Validate required fields
        if (
            !calendar_id_external || 
            !date_of_holiday || 
            !holiday_class || 
            !holiday_code ||
            !holiday_category) 
            {
            const errorMessage = "Missing required fields! " 
            return sendErrorResponse(res,errorMessage , errorMessage);
        }

        // Check for existing active record
        const existingRecord = await performQuery(
          `SELECT * FROM ${tables.holiday_assignments} WHERE calendar_id_external = ? AND date_of_holiday = ? AND holiday_code = ? AND is_active = 1`,
          [calendar_id_external, date_of_holiday, holiday_code]
        );
        if (existingRecord.length > 0) {
          return sendErrorResponse(res, "Holiday Assignments for the given calendar and date already exists", "Holiday Assignments for the given calendar and date already exists");
        }

        // Check if Picklist values exist for holiday_class and holiday_category
        const checkHolidayClass = await performQuery(
          `SELECT * FROM ${tables.picklist_master} WHERE (picklist_id = 16 and id = ?) AND is_deleted = 2`,
          [holiday_class]
        );
        if (checkHolidayClass.length === 0) {
          return sendErrorResponse(res, "Invalid holiday class", "Invalid holiday class");
        }
        const checkHolidayCategory =  await performQuery(
          `SELECT * FROM ${tables.picklist_master} WHERE (picklist_id = 17 and id = ?) AND is_deleted = 2`,
          [holiday_category]
        );
        if (checkHolidayCategory.length === 0) {
          return sendErrorResponse(res, "Invalid holiday category", "Invalid holiday category");
        }
        
        // check if calendar_id_external exists in holiday_calendar table
        const checkCalendar =  await performQuery(
          `SELECT * FROM ${tables.holiday_calendar} WHERE calendar_id_external = ? AND is_active = 1`,
            [calendar_id_external]
        );
        if (checkCalendar.length === 0) {
          return sendErrorResponse(res, "Invalid calendar ID", "Invalid calendar ID");
        }

        // Check if holiday_code exists in holidays table
        const checkHolidayCode =  await performQuery(
          `SELECT * FROM ${tables.holidays} WHERE holiday_code = ? AND is_active = 1`,
            [holiday_code]
        );
        if (checkHolidayCode.length === 0) {
          return sendErrorResponse(res, "Invalid holiday code", "Invalid holiday code");
        }

        // Check is country of calendar and holiday match
        if (checkCalendar[0].country !== checkHolidayCode[0].country) {
          return sendErrorResponse(res, "Country mismatch between calendar and holiday", "Country mismatch between calendar and holiday");
        }
        
        // Get current system time
        const systemTime = await getSystemTime();
        const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");
    
        const result = await performQuery(
          `INSERT INTO ${tables.holiday_assignments} SET ?`,
          {
            sequence_no: 1,
            calendar_id_external,
            date_of_holiday,
            holiday_class,
            holiday_code,
            holiday_category,
            created_by: req?.user?.id,
            created_at: currentTime,
            updated_by: req?.user?.id,
            updated_at: currentTime,
          }
        );
    
        return sendResponse(res, {}, "Holiday Assignments created successfully", 200);
    
    } catch (error) {
        console.error("Error While Creating Holiday Assignments", error);
        return sendErrorResponse(res, error, "Error while creating holiday Assignments");
    }
};

// --> Get Holidays Assignments
module.exports.getHolidayAssignments = async (req, res) => {
    try {
    const { id } = req.query;

    let query = `
        SELECT 
            ha.id, 
            ha.calendar_id_external, hc.name_en AS calendar_name, hc.country AS calendar_country,
            ha.date_of_holiday, 
            ha.holiday_class, pcm.picklist_option AS holiday_class_value,
            ha.holiday_code, h.name AS holiday_name,
            ha.holiday_category, pcm2.picklist_option AS holiday_category_value
        FROM ${tables.holiday_assignments} ha
        LEFT JOIN ${tables.holiday_calendar} hc ON ha.calendar_id_external = hc.calendar_id_external AND hc.is_active = 1
        LEFT JOIN ${tables.picklist_master} pcm ON ha.holiday_class = pcm.id AND pcm.picklist_id = 16 AND pcm.is_deleted = 2
        LEFT JOIN ${tables.picklist_master} pcm2 ON ha.holiday_category = pcm2.id AND pcm2.picklist_id = 17 AND pcm2.is_deleted = 2
        LEFT JOIN ${tables.holidays} h ON ha.holiday_code = h.holiday_code AND h.is_active = 1
        WHERE ha.is_active = 1
    `;

    if (id) query += ` AND ha.id = ${id}`;
    query += ` ORDER BY ha.id DESC`;
    const result = await performQuery(query);
    return sendResponse(res, result, "Holiday Assignments fetched successfully", 200);

  } catch (error) {
    console.error("Error While Fetching Holiday Assignments", error);
    return sendErrorResponse(res, error, "Error While Fetching Holiday Assignments");
  }
};

// --> Update Holidays Assignments
module.exports.updateHolidayAssignments = async (req, res) => {
    try {
        const {
            id,
            date_of_holiday,
            holiday_class,
            holiday_code,
            holiday_category,
        } = req.body;
    
        if (!id || !date_of_holiday || !holiday_class || !holiday_code || !holiday_category) {
            const errorMessage = "Missing required fields! "
            return sendErrorResponse(res,errorMessage , errorMessage);
        }
    
        // Check existing record
        const existingRecord = await performQuery(
            `SELECT * FROM ${tables.holiday_assignments} WHERE id = ? AND is_active = 1`,
            [id]
        );
        if (existingRecord.length === 0) {
            return sendErrorResponse(res, "Holiday Assignments not found", "Holiday Assignments not found");
        }
        const calendar_id_external = existingRecord[0].calendar_id_external;

        // Check for date of holiday with same calendar_id_external (excluding current record)
        const duplicateRecord = await performQuery(
          `SELECT * FROM ${tables.holiday_assignments} WHERE calendar_id_external = ? AND date_of_holiday = ? AND holiday_code = ? AND is_active = 1 AND id <> ?`,
          [calendar_id_external, date_of_holiday, holiday_code, id]
        );
        if (duplicateRecord.length > 0) {
          return sendErrorResponse(res, "Holiday Assignments for the given calendar and date already exists", "Holiday Assignments for the given calendar and date already exists");
        }

        // Check if Picklist values exist for holiday_class and holiday_category
        const checkHolidayClass = await performQuery(
          `SELECT * FROM ${tables.picklist_master} WHERE (picklist_id = 16 and id = ?) AND is_deleted = 2`,
          [holiday_class]
        );
        if (checkHolidayClass.length === 0) {
          return sendErrorResponse(res, "Invalid holiday class", "Invalid holiday class");
        }
        const checkHolidayCategory =  await performQuery(
          `SELECT * FROM ${tables.picklist_master} WHERE (picklist_id = 17 and id = ?) AND is_deleted = 2`,
          [holiday_category]
        );
        if (checkHolidayCategory.length === 0) {
          return sendErrorResponse(res, "Invalid holiday category", "Invalid holiday category");
        }
        
        // Check if same holiday_code exists for same calendar in holidays_assignments table (excluding current record)
        const checkCalendar =  await performQuery(
          `SELECT * FROM ${tables.holiday_assignments} WHERE calendar_id_external = ? AND holiday_code = ? AND is_active = 1 and id <> ?`,
            [calendar_id_external, holiday_code, id]
        );
        if (checkCalendar.length > 0) {
            return sendErrorResponse(res, "Holiday Already Exist in this calendar", "Holiday Already Exist in this calendar");
        }

        // Get current system time
        const systemTime = await getSystemTime();
        const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");
    
        // Deactivate existing record to maintain history
        await performQuery(
            `UPDATE ${tables.holiday_assignments} SET ? WHERE id = ?`,
            [{
                is_active: 2,
                updated_by: req?.user?.id,
                updated_at: currentTime
            },id]
        );
    
        // Insert new record with updated details
        const result = await performQuery(
            `INSERT INTO ${tables.holiday_assignments} SET  ?`,
            {
                sequence_no: Number(existingRecord[0].sequence_no) + 1,
                calendar_id_external,
                date_of_holiday,
                holiday_class,
                holiday_code,
                holiday_category,
                created_by: req?.user?.id,
                created_at: currentTime,
                updated_by: req?.user?.id,
                updated_at: currentTime
            }
        );
    
        return sendResponse(res, {}, "Holiday Assignments updated successfully", 200);
    
    } catch (error) {
        console.log("Error While Updating Holiday Assignments", error);
        return sendErrorResponse(res, error, "Error while updating holiday Assignments");
    }
};
